/******************************************************************************
    SG_Touch.c        Touch Panel testing. 
    Copyright:        Open-Source, Jan2011, written by www.RomanBlack.com
    MCU:              P18F8527
    Dev.Board:        SmartGLCD (MikroE SmartGLCD 240x128 module)
    Oscillator:       HSPLL 32.0 MHz (8MHz xtal)
    Compiler:         MikroC v7.0 (or MikroC PRO 4.0)

  This demonstrates the GLCD and Touch panel and displays X and Y coords
  on the GLCD as two text numbers. It uses 4 very simple and fast functions
  that set the TouchPanel and ADC to the right pins and read the ADC.
  It only reads X or Y from the touchpanel at one time, the user must test
  both (if required) to get an XY touch position. These TPfast functions
  are optimised for simplicity and speed, and will be useful if you are doing
  time-critical applications that occasionally need to get a touch panel
  reading. There is no calibration, the X and Y values are ADC values 0-255.
******************************************************************************/
// Global Variables

unsigned char tp_x;     // will hold results of Touch Panel adc values
unsigned char tp_y;
unsigned char text[4];  // holds 3 digit number + NULL, for text display

// TPfast function definitions
void TPfast_SetX(void);   // function to set X axis for next read
void TPfast_SetY(void);   // function to set X axis for next read
void TPfast_StartRead(void);      // function to start adc read from TouchPanel
unsigned char TPfast_Read(void);  // function to read adc result from TouchPanel

#include "T6963C_MikroC.h"      // include this if using MikroC (older)
//#include "T6963C_MikroC_PRO.h"  // include this if using MikroC PRO

//=============================================================================
//  MAIN
//=============================================================================
void main()
{
  //-------------------------------------------------------
  // setup PIC 18F8527 for SmartGLCD pins
  CMCON = 0x07;        // turn off comparators (make all pins digital)
  ADCON0 = 0b00000001;  // ADC module on
  ADCON1 = 0b00001101;  // AN0,AN1 are adc inputs, 0v-5v range
  ADCON2 = 0b00110010;  // ADC result left justified (0-255 range)

  LATA =  0b00000000;
  TRISA = 0b00000011;   // RA0,RA1 analog inputs (TP)
  LATC =  0b00000110;   // LEDs off at start
  TRISC = 0b00000000;   // C1, C2 backlight LED
  LATG =  0b00000001;   // LED off at start
  TRISG = 0b00000000;   // G0 backlight LED
  
  LATJ  = 0b00000000;   // RJ6=FS (1=font6 0=font8), RJ5=MD
  TRISJ = 0b00000000;   // GLCD control port

  BacklightRed    = 1;     // control the GLCD backlight leds; 0=on, 1=off
  BacklightGreen  = 0;
  BacklightBlue   = 1;

  //-------------------------------------------------------
  // startup delay, let the PSU voltage stabilise etc.
  Delay_ms(10);

  // Initialize T6963C GLCD
  //T6963C_init(240, 128, 8);   // init for MikroC PRO version
  T6963C_init(240, 128, 8, &PORTH, &PORTJ, 2, 1, 0, 4); // init for MikroC version
  T6963C_graphics(1);       // graphics mode = on
  T6963C_text(1);           // text mode = on (now both are on)
  T6963C_cursor(1);         // cursor and blink don't seem to work?
  T6963C_cursor_blink(1);

  //-------------------------------------------------------
  // draw stuff on GLCD that remains constant
  // draw an outline rectangle around entire screen area
  T6963C_rectangle(0, 0, 239, 127, T6963C_WHITE);   // is 240x128 (needs -1)

  // write the text that will not change
  T6963C_Write_Text("SmartGLCD TouchPanel testing", 1, 1, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("using the TPfast() functions", 1, 2, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("X=", 11, 6, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("Y=", 11, 9, T6963C_ROM_MODE_OR);
 
  //-------------------------------------------------------
  // main running loop (repeat 10 times a second)
  while(1)
  {
    // get TouchPanel X value
    TPfast_SetX();       
    Delay_uS(500);
    TPfast_StartRead();   // start TP adc conversion
    tp_x = TPfast_Read();

    // get TouchPanel Y value
    TPfast_SetY();       
    Delay_uS(500);
    TPfast_StartRead();   // start TP adc conversion
    tp_y = TPfast_Read();

    // format and display TP X value to screen
    ByteToStr(tp_x,text);
    T6963C_Write_Text(text, 14, 6, T6963C_ROM_MODE_OR);

    // format and display TP Y value to screen
    ByteToStr(tp_y,text);
    T6963C_Write_Text(text, 14, 9, T6963C_ROM_MODE_OR);

    Delay_mS(100);
  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//  TPFAST SETX
//=============================================================================
void TPfast_SetX(void)
{
  //-------------------------------------------------------
  // RomanBlack's high performance TouchPanel read functions for
  // SmartGLCD240x128 module; Jan 2011 - open source.
  // Call this function to set up the TP to read the X axis.
  // Note! If you have changed axis, there must be a delay of at least
  // 500uS before reading the new selected axis (to allow ADC to settle).
  //-------------------------------------------------------
  // set the PIC pins so it can read the TP X axis
  LATA.F5 = 0;    // set DRIVEA and DRIVEB pins to TP
  LATA.F2 = 1;   

  // set the ADC to read from the now floating pin (TP BOTTOM on RA0)
  ADCON0 = 0b00000001;  // adc channel AN0 is selected
}
//-----------------------------------------------------------------------------

//=============================================================================
//  TPFAST SETY
//=============================================================================
void TPfast_SetY(void)
{
  //-------------------------------------------------------
  // RomanBlack's high performance TouchPanel read functions for
  // SmartGLCD240x128 module; Jan 2011 - open source.
  // Call this function to set up the TP to read the Y axis.
  // Note! If you have changed axis, there must be a delay of at least
  // 500uS before reading the new selected axis (to allow ADC to settle).
  //-------------------------------------------------------
  // set the PIC pins so it can read the TP Y axis
  LATA.F2 = 0;    // set DRIVEA and DRIVEB pins to TP
  LATA.F5 = 1;    //

  // set the ADC to read from the now floating pin (TP LEFT on RA1)
  ADCON0 = 0b00000101;  // adc channel AN1 is selected
}
//-----------------------------------------------------------------------------

//=============================================================================
//  TPFAST STARTREAD
//=============================================================================
void TPfast_StartRead(void)
{
  //-------------------------------------------------------
  // RomanBlack's high performance TouchPanel read functions for
  // SmartGLCD240x128 module; Jan 2011 - open source.
  // Call this function to start an ADC read of the TP.
  //-------------------------------------------------------
  // read either X or Y axis of the touchpanel, based on
  // the TP axis which was last set up.
  ADCON0.F1 = 1;  // initiate a ADC read
}
//-----------------------------------------------------------------------------

//=============================================================================
//  TPFAST READ
//=============================================================================
unsigned char TPfast_Read(void)
{
  //-------------------------------------------------------
  // RomanBlack's high performance TouchPanel read functions for
  // SmartGLCD240x128 module; Jan 2011 - open source.
  // Call this function to get the ADC value of the TP.
  //-------------------------------------------------------
  // get either X or Y axis of the touchpanel, based on
  // the TP axis which was last set up.
  
  // wait for adc conversion to finish if it is still going!
  while(ADCON0.F1) continue;
  
  // then get the adc result (0-255) and return it
  return(ADRESH); 
}
//-----------------------------------------------------------------------------


