/******************************************************************************
    SG_Touch_Int.c    TouchPanel reading using interrupt. 
    Copyright:        Open-Source, Jan2011, written by www.RomanBlack.com
    MCU:              P18F8527
    Dev.Board:        SmartGLCD (SmartGLCD 240x128 module)
    Oscillator:       HSPLL 32.0 MHz (8MHz xtal)
    Compiler:         MikroC v7.0 (or MikroC PRO 4.0)

  This demonstrates the GLCD and Touch Panel and displays X and Y coords
  on the GLCD as two text numbers. It uses a simple and very fast interrupt
  to control all the fast TouchPanel "TPfast" functions so that the user
  always has X and Y available from the TP without ever needing to test it.
  NOTE!! The TPfast() functions used to read the TouchPanel have been moved
  to their own file called "TPfast.c"
******************************************************************************/
// Global Variables

unsigned char tpwork_x; // holds temporary working values of TP X and Y
unsigned char tpwork_y;
unsigned char tp_x;     // will hold final results of Touch Panel adc values
unsigned char tp_y;
unsigned char tp_xy_locked; // locks XY values so the int can't change them
unsigned char text[4];      // holds 3 digit number + NULL, for text display
unsigned char int_count;    // counts sequence of TMR0 interrupts

#include "T6963C_MikroC.h"      // include this if using MikroC (older)
//#include "T6963C_MikroC_PRO.h"  // include this if using MikroC PRO

#include "TPfast.c"     // needed for my TouchPanel functions 

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void interrupt()
{
  //-------------------------------------------------------
  // TMR0 overflow interrupt; prescale 1:16 = 409uS, (PIC 32MHz HSPLL)
  // The TMR0 int performs 4 sequential functions (each is 1 int);
  //  00 = start read X
  //  01 = read TP X, then set TP to Y,
  //  10 = start read Y
  //  11 = read TP Y, then set TP to X, and output X Y
  // These cases are handled with bit testing for highest speed so this
  // interrupt is very fast and efficient (although it looks messy).
  //-------------------------------------------------------
  if(!int_count.F1)     // cases 00 and 01
  {
    if(!int_count.F0)   // case 00
    {
      TPfast_StartRead();         // start read X
    }
    else                // case 01
    {
      tpwork_x = TPfast_Read();   // read TP X
      TPfast_SetY();              // then set TP to Y
    }
  }
  else                  // cases 10 and 11
  {
    if(!int_count.F0)   // case 10
    {
      TPfast_StartRead();       // start read Y
    }
    else                // case 11
    {
      tpwork_y = TPfast_Read();   // read TP Y
      TPfast_SetX();              // then set TP to X
      // now output XY values if not locked!
      if(!tp_xy_locked)
      {
        if(tpwork_y > 12)     // if TouchPanel is pressed y>12
        {
          tp_x = tpwork_x;    // load XY into user variables
          tp_y = tpwork_y;
        }
        else        // else TP was not pressed!
        {
          tp_x = 0;    // clear XY to indicate not pressed
          tp_y = 0;
        }
      }
    }
  }  
  int_count++;          // inc sequence
  INTCON.TMR0IF = 0;    // clear the int flag before exit
}
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


//=============================================================================
//  MAIN
//=============================================================================
void main()
{
  //-------------------------------------------------------
  // setup PIC 18F8527 for SmartGLCD pins
  CMCON = 0x07;        // turn off comparators (make all pins digital)
  ADCON0 = 0b00000001;  // ADC module on
  ADCON1 = 0b00001101;  // AN0,AN1 are adc inputs, 0v-5v range
  ADCON2 = 0b00110010;  // ADC result left justified (0-255 range)

  LATA =  0b00000000;
  TRISA = 0b00000011;   // RA0,RA1 analog inputs (TP)
  LATC =  0b00000110;   // LEDs off at start
  TRISC = 0b00000000;   // C1, C2 backlight LED
  LATG =  0b00000001;   // LED off at start
  TRISG = 0b00000000;   // G0 backlight LED
  
  LATJ  = 0b00000000;   // RJ6=FS (1=font6 0=font8), RJ5=MD
  TRISJ = 0b00000000;   // GLCD control port

  T0CON = 0b11000011;   // TMR0 enabled, 8bit, 1:16 prescaler

  BacklightRed    = 1;     // control the GLCD backlight leds; 0=on, 1=off
  BacklightGreen  = 0;
  BacklightBlue   = 1;

  //-------------------------------------------------------
  // startup delay, let the PSU voltage stabilise etc.
  Delay_ms(10);

  // Initialize T6963C GLCD
  //T6963C_init(240, 128, 8);   // init for MikroC PRO version
  T6963C_init(240, 128, 8, &PORTH, &PORTJ, 2, 1, 0, 4); // init for MikroC version
  T6963C_graphics(1);       // graphics mode = on
  T6963C_text(1);           // text mode = on (now both are on)
  T6963C_cursor(1);         // cursor and blink don't seem to work?
  T6963C_cursor_blink(1);

  //-------------------------------------------------------
  // draw stuff on GLCD that remains constant
  // draw an outline rectangle around entire screen area
  T6963C_rectangle(0, 0, 239, 127, T6963C_WHITE);   // is 240x128 (needs -1)

  // draw a small box for the indicator
  T6963C_box(54-1,64-1,54+7+1,64+7+1, T6963C_WHITE);

  // write the text that will not change
  T6963C_Write_Text("SmartGLCD TPfast() testing,", 1, 1, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("using interrupt for auto XY", 1, 2, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("X=", 11, 6, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("Y=", 11, 9, T6963C_ROM_MODE_OR);
 
  //-------------------------------------------------------
  // load any variables before starting
  int_count = 0;  
  tpwork_x = 0;
  tpwork_y = 0;
  tp_x = 0;
  tp_y = 0;
  tp_xy_locked = 0;
  
  // turn TMR0 interrupt on
  INTCON.GIE = 1;
  INTCON.TMR0IE = 1;
  
  // main running loop (repeat 10 times a second)
  while(1)
  {
    //---------------------------------------------
    // do whatever here (TouchPanel is being read automatically)
    Delay_mS(100);

    //---------------------------------------------
    // now grab the current TouchPanel XY and dislay it
    tp_xy_locked = 1;   // lock the current TP XY values so int can't change them

    // draw a little indicator box if TP was pressed or not
    if(tp_x)  T6963C_box(54,64,54+7,64+7, T6963C_WHITE);  
    else      T6963C_box(54,64,54+7,64+7, T6963C_BLACK);  

    // format and display TP X value
    ByteToStr(tp_x,text);
    T6963C_Write_Text(text, 14, 6, T6963C_ROM_MODE_OR);

    // format and display TP Y value
    ByteToStr(tp_y,text);
    T6963C_Write_Text(text, 14, 9, T6963C_ROM_MODE_OR);

    tp_xy_locked = 0;    // unlcok the XY so int can write to them again
    //---------------------------------------------    
  }
}
//-----------------------------------------------------------------------------







