/******************************************************************************
    SG_40x16_text.c   Test of using 40x16 text mode for more characters...
    Copyright:        Open-Source, Jan2011, written by www.RomanBlack.com
    MCU:              P18F8527
    Dev.Board:        SmartGLCD (SmartGLCD 240x128 module)
    Oscillator:       HSPLL 32.0 MHz (8MHz xtal)
    Compiler:         MikroC v7.0 (or MikroC PRO 4.0)

  This project sets the SmartGLCD to 40x16 text character mode and then
  draws a screen full of text and some graphics items to check that the
  GLCD is working ok in this 40x16 mode.
  It also uses my easy PWM RGB LED backlight code to set the backlight
  to a nice blue colour.
******************************************************************************/
// Global Variables
// defines used for the LED PWM system
unsigned char int_count;
unsigned char pwm_red;    // pwm values for display
unsigned char pwm_green;
unsigned char pwm_blue;

unsigned char i;


#include "T6963C_MikroC.h"      // include this if using MikroC (older)
//#include "T6963C_MikroC_PRO.h"  // include this if using MikroC PRO

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void interrupt()
{
  //-------------------------------------------------------
  // TMR0 overflow interrupt; prescale 1:1
  // The TMR0 int performs PWM on the backlight RGB LED.
  // This is a fast smiple int, basically it just compares the int_count
  // to the PWM value for each colour.
  // TMR0L bit7 is forced hi, so int happens every 128+2 ticks
  // (total PWM cycle is about 240 Hz)
  //-------------------------------------------------------
  if(!int_count)    // when int_count==0, turn off all LEDs
  {
    BacklightRed   = 1;    // 1 is off 
    BacklightGreen = 1;  
    BacklightBlue  = 1;  
  }
  else       // for 1-255, some LEDs might be on
  {
    // turn each LED on when reaches pwm value
    if(int_count == pwm_red)   BacklightRed   = 0;
    if(int_count == pwm_green) BacklightGreen = 0;
    if(int_count == pwm_blue)  BacklightBlue  = 0;
  }
  int_count--;          // inc sequence (must count DOWN)
  TMR0L.F7 = 1;         // make TMR0 roll over every 128+2 ticks
  INTCON.TMR0IF = 0;    // clear the int flag before exit
}
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//=============================================================================
//  BLANK
//=============================================================================
void blank(void)
{
  //-------------------------------------------------------

}
//-----------------------------------------------------------------------------


//=============================================================================
//  MAIN
//=============================================================================
void main()
{
  //-------------------------------------------------------
  // setup PIC 18F8527 for SmartGLCD pins
  CMCON = 0x07;        // turn off comparators (make all pins digital)
  ADCON0 = 0b00000001;  // ADC module on
  ADCON1 = 0b00001101;  // AN0,AN1 are adc inputs, 0v-5v range
  ADCON2 = 0b00110010;  // ADC result left justified (0-255 range)

  LATA =  0b00000000;
  TRISA = 0b00000011;   // RA0,RA1 analog inputs (TP)
  LATC =  0b00000110;   // LEDs off at start
  TRISC = 0b00000000;   // C1, C2 backlight LED
  LATG =  0b00000001;   // LED off at start
  TRISG = 0b00000000;   // G0 backlight LED
  
  // NOTE!! this line below sets the GLCD to small font (6x8 pixels)
  LATJ  = 0b01000000;   // RJ6=FS (1=font6 0=font8), RJ5=MD
  TRISJ = 0b00000000;   // GLCD control port

  BacklightRed    = 1;     // control the GLCD backlight leds; 0=on, 1=off
  BacklightGreen  = 1;     // start with LEDs off
  BacklightBlue   = 1;

  T0CON = 0b11000000;   // TMR0 enabled, 8bit, 1:1 prescaler

  //-------------------------------------------------------
  // startup delay, let the PSU voltage stabilise etc.
  Delay_ms(10);

  // Initialize T6963C GLCD
  // NOTE!! The init() function below sets the GLCD addressing to 6x8 font (6pixels per horixontal byte)
  //T6963C_init(240, 128, 6);   // init for MikroC PRO version
  T6963C_init(240, 128, 6, &PORTH, &PORTJ, 2, 1, 0, 4); // init for MikroC version
  T6963C_graphics(1);       // graphics mode = on
  T6963C_text(1);           // text mode = on (now both are on)
  T6963C_cursor(0);         // cursor = off

  // set a nice colour blend on the RGB LED backlight
  pwm_red   = 60;    
  pwm_green = 120;
  pwm_blue  = 200;
  int_count = 0;

  // turn TMR0 interrupt on (used for LED PWM)
  INTCON.GIE = 1;
  INTCON.TMR0IE = 1;

  //-------------------------------------------------------
  // draw stuff on GLCD that remains constant.
  // write the text that will not change
  T6963C_Write_Text("This is a test of 40 x 16 character mode", 0, 0, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("This is a test of 40 x 16 character mode", 0, 1, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("This is a test of 40 x 16 character mode", 0, 2, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("This is a test of 40 x 16 character mode", 0, 3, T6963C_ROM_MODE_OR);
  
  T6963C_Write_Text("This is a test of 40 x 16 character mode", 0, 4, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("This is a test of 40 x 16 character mode", 0, 5, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("This is a test of 40 x 16 character mode", 0, 6, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("This is a test of 40 x 16 character mode", 0, 7, T6963C_ROM_MODE_OR);

  // also draw a simple graphic bargraph object to show graphics are
  // still working ok in the new text mode.
  T6963C_rectangle(0,68,239,68+22, T6963C_WHITE);   // long bargraph outer frame
  for(i=3; i<239-3; i+=2)
  {
    T6963C_Line(i,68+4,i,68+22-4,T6963C_WHITE);     // fill it with lines
  }

  T6963C_Write_Text("This is a test of 40 x 16 character mode", 0, 12, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("This is a test of 40 x 16 character mode", 0, 13, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("This is a test of 40 x 16 character mode", 0, 14, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("This is a test of 40 x 16 character mode", 0, 15, T6963C_ROM_MODE_OR);
  
  //-------------------------------------------------------

  // main running loop (repeat 10 times a second)
  while(1)
  {
    // just loop and do nothing
  }
}
//-----------------------------------------------------------------------------





