/******************************************************************************
    SG_fast_GLCD.c    First attempt to speed up the graphics drawing!
    Copyright:        Open-Source, Jan2011, written by www.RomanBlack.com
    MCU:              P18F8527
    Dev.Board:        SmartGLCD (SmartGLCD 240x128 module)
    Oscillator:       HSPLL 32.0 MHz (8MHz xtal)
    Compiler:         MikroC v7.0 (or MikroC PRO 4.0)

  I wrote 3 very simple functions to allow data bytes and command bytes to
  be written directly to the GLCD. Then I tested these functions for speed
  drawing a line across the screen and vertically down the screen
  and compared the speed to the MikroE standard line drawing functions.
  TMR1 was set to 1uS per tick, and it was used to time the events
  in actual uS and display on the screen.
******************************************************************************/
// Global Variables
unsigned char i;          // used for loops

unsigned int etime;       // used to time events in uS

unsigned char txt[6];     // holds number for displaying on screen as text

unsigned char RSG_byte;   // used by my functions

unsigned char add_hi;
unsigned char add_lo;

#include "T6963C_MikroC.h"      // include this if using MikroC (older)
//#include "T6963C_MikroC_PRO.h"  // include this if using MikroC PRO



//=============================================================================
//  RomanSG_CHECK_STATUS 
//=============================================================================
void RomanSG_Check_Status(void)
{
  //-------------------------------------------------------
  // Written for SmartGLCD module with T6963C GLCD only!
  // This is a test of a very fast function to check if the
  // GLCD is busy. It returns when the GLCD is free.
  // PORTJ = Res  CE   WR  RD  C/D  
  // PORTJ   4    3    2   1    0  
  //-------------------------------------------------------
  unsigned char read_val;
  TRISH = 0b11111111;   // set GLCD data port H to all inputs
  LATJ |= 0b00001111;   // ce wr rd cd
  LATJ.F1 = 0;          // read is low

  while(1)              // loop here until GLCD is free
  {
    asm nop;     
    LATJ.F3 = 0;        // CE low
    asm nop;     
    read_val = PORTH;   // get the status byte
    LATJ.F3 = 1;        // CE high again
    if((read_val & 0x03) == 0x03) break;     // break if free
  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//  RomanSG_SEND_DATA
//=============================================================================
void RomanSG_Send_Data(void)
{
  //-------------------------------------------------------
  // Written for SmartGLCD module with T6963C GLCD only!
  // This is a test of a very fast function to write one data byte
  // to the GLCD. The data byte is already in RSG_byte, for best speed.
  // PORTJ = Res  CE   WR  RD  C/D  
  // PORTJ   4    3    2   1    0  
  //-------------------------------------------------------
  RomanSG_Check_Status();   // make sure GLCD is free first

  TRISH = 0b00000000;   // set GLCD data port H to all outputs
  LATH = RSG_byte;      // load the byte ready to the data port

  LATJ |= 0b00001111;   // ce wr rd cd
  LATJ &= 0b11111010;   // wr and cd are made low
  asm nop;
  LATJ.F3 = 0;          // CE low
  asm nop;
  LATJ.F3 = 1;          // CE high again
  asm nop;

  LATJ |= 0b00001111;   // ce wr rd cd
}
//-----------------------------------------------------------------------------


//=============================================================================
//  RomanSG_SEND_COMMAND
//=============================================================================
void RomanSG_Send_Command(void)
{
  //-------------------------------------------------------
  // Written for SmartGLCD module with T6963C GLCD only!
  // This is a test of a very fast function to write one command byte
  // to the GLCD. The command byte is already in RSG_byte, for best speed.
  // PORTJ = Res  CE   WR  RD  C/D  
  // PORTJ   4    3    2   1    0  
  //-------------------------------------------------------
  RomanSG_Check_Status();   // make sure GLCD is free first

  TRISH = 0b00000000;   // set GLCD data port H to all outputs
  LATH = RSG_byte;      // load the byte ready to the data port

  LATJ |= 0b00001111;   // ce wr rd cd
  LATJ.F2 = 0;          // write is low
  asm nop;
  LATJ.F3 = 0;          // CE low
  asm nop;
  LATJ.F3 = 1;          // CE high again
  asm nop;

  LATJ |= 0b00001111;   // ce wr rd cd
  LATJ.F3 = 0;          // ce low (MikroE functions need this left low on exit!)
}
//-----------------------------------------------------------------------------


//=============================================================================
//  MAIN
//=============================================================================
void main()
{
  //-------------------------------------------------------
  // setup PIC 18F8527 for SmartGLCD pins
  CMCON = 0x07;        // turn off comparators (make all pins digital)
  ADCON0 = 0b00000001;  // ADC module on
  ADCON1 = 0b00001101;  // AN0,AN1 are adc inputs, 0v-5v range
  ADCON2 = 0b00110010;  // ADC result left justified (0-255 range)

  LATA =  0b00000000;
  TRISA = 0b00000011;   // RA0,RA1 analog inputs (TP)
  LATC =  0b00000110;   // LEDs off at start
  TRISC = 0b00000000;   // C1, C2 backlight LED
  LATG =  0b00000001;   // LED off at start
  TRISG = 0b00000000;   // G0 backlight LED
  
  LATJ  = 0b01000000;   // RJ6=FS (1=font6 0=font8), RJ5=MD
  TRISJ = 0b00000000;   // GLCD control port

  BacklightRed    = 1;     // control the GLCD backlight leds; 0=on, 1=off
  BacklightGreen  = 0;     // green ON
  BacklightBlue   = 1;

  T1CON = 0b10110001;   // TMR1 on 16bit, 1:8 prescaler, used for time testing

  //-------------------------------------------------------
  // Initialize T6963C GLCD
  //T6963C_init(240, 128, 6);   // init for MikroC PRO version
  T6963C_init(240, 128, 6, &PORTH, &PORTJ, 2, 1, 0, 4); // init for MikroC version
  T6963C_graphics(1);       // graphics mode = on
  T6963C_text(1);           // text mode = on (now both are on)
  T6963C_cursor(0);         // cursor = off

  Delay_mS(300);

  //-------------------------------------------------------
  // draw stuff on GLCD.
  //-------------------------------------------------------
  
  // do a time test of the MikroE horiz line, this draws a line
  // across the screen and times it in exact uS.
  T6963C_Write_Text("Line drawn with MikroE Line() function", 0, 0, T6963C_ROM_MODE_OR); 

  TMR1H=0;
  TMR1L=0;              // clear TMR1 here
  T6963C_line(0,16,239,16,T6963C_WHITE);    // Draw a MikroE line
  etime = TMR1L;        // read TMR1 here
  etime += (TMR1H << 8);

  WordToStr(etime,txt);    // get the time as text
  T6963C_Write_Text("Time:       uS", 0, 4, T6963C_ROM_MODE_OR);   // display time in uS
  T6963C_Write_Text(txt, 6, 4, T6963C_ROM_MODE_OR);   // display time in uS

  //-------------------------------------------------------
  // now do a time test using my line bytes system, this is a "best case"
  // where all it does is draw bytes (each byte is 6 black pixels)
  // across the screen to make a horizontal line. This is horrible code but
  // it is about as fast as this GLCD can ever get, which is the test.

  T6963C_Write_Text("Line drawn with my functions", 0, 10, T6963C_ROM_MODE_OR); 

  TMR1H=0;
  TMR1L=0;          // clear TMR1 here

  // set first graphic byte to write to (each pixel row across is 40 bytes)
  // y = 40*8*textline = 40*8*12 = 0x0F00
  // this should be put in a function later! RomanSG_Set_Address(address)
  RSG_byte = 0x00;        // low address in graphic ram
  RomanSG_Send_Data();
  RSG_byte = 0x0F;        // hi address in graphic ram
  RomanSG_Send_Data();
  RSG_byte = 0x24;        // command to set ram address pointer
  RomanSG_Send_Command();

  // now loop and draw 40 bytes into graphic ram
  for(i=0; i<40; i++)
  {
    RSG_byte = 0x3F;        // 0b00111111 (6 black pixels to write to graphic ram)
    RomanSG_Send_Data();
    RSG_byte = 0xC0;        // command to write byte to graphic ram and increment ram pointer
    RomanSG_Send_Command();
  }
  etime = TMR1L;        // read TMR1 here
  etime += (TMR1H << 8);
  
  WordToStr(etime,txt);      // get the time as text
  T6963C_Write_Text("Time:       uS", 0, 14, T6963C_ROM_MODE_OR);   // display time in uS
  T6963C_Write_Text(txt, 6, 14, T6963C_ROM_MODE_OR);

  //-------------------------------------------------------
  // delay between test pages
  Delay_mS(3000);
  
  // clear the screen
  T6963C_grFill(0);     // erase graphics
  T6963C_txtFill(0);    // erase text

  //-------------------------------------------------------
  
  
  // do a time test of the MikroE vertical line
  T6963C_Write_Text("Vert line with MikroE Line() function", 0, 0, T6963C_ROM_MODE_OR); 

  TMR1H=0;
  TMR1L=0;              // clear TMR1 here
  T6963C_line(29,0,29,127,T6963C_WHITE);    // Draw a MikroE line
  etime = TMR1L;        // read TMR1 here
  etime += (TMR1H << 8);

  WordToStr(etime,txt);    // get the time as text
  T6963C_Write_Text("Time:       uS", 0, 4, T6963C_ROM_MODE_OR);   // display time in uS
  T6963C_Write_Text(txt, 6, 4, T6963C_ROM_MODE_OR);   // display time in uS

  //-------------------------------------------------------
  
  // and test manual writing of a vertical line by drawing to graphcis ram...
  
  T6963C_Write_Text("Vert line with my functions", 0, 10, T6963C_ROM_MODE_OR); 
  TMR1H=0;
  TMR1L=0;          // clear TMR1 here

  add_lo = 31;
  add_hi = 0;

  // now loop and draw 120 vertical pixels, top down
  for(i=0; i<128; i++)
  {
    // set graphics ram address for this byte
    RSG_byte = add_lo;      // low address in graphic ram
    RomanSG_Send_Data();
    RSG_byte = add_hi;      // hi address in graphic ram
    RomanSG_Send_Data();
    RSG_byte = 0x24;        // command to set ram address pointer
    RomanSG_Send_Command();

    // and make a single black pixel bit in that byte (bit 3 011)
    RSG_byte = 0b11111011;   // command to set a pixel; 1111 colour ppp
    RomanSG_Send_Command();
    
    /*
    T6963C_writeData(add_lo);   // MikroC command functions were twice as slow
    T6963C_writeData(add_hi);
    T6963C_writeCommand(0x24);
    
    T6963C_writeCommand(0b11111011);
    */

    // manually calc the address of the next byte below, is +=40 bytes.
    // manually handle the 16bit address for speed (messy!)
    if(add_lo >= (256-40)) add_hi ++; 
    add_lo += 40;   // is a 16bit +=40
  }
  etime = TMR1L;        // read TMR1 here
  etime += (TMR1H << 8);
  
  WordToStr(etime,txt);      // get the time as text
  T6963C_Write_Text("Time:       uS", 0, 14, T6963C_ROM_MODE_OR);   // display time in uS
  T6963C_Write_Text(txt, 6, 14, T6963C_ROM_MODE_OR);
  
  
  //-------------------------------------------------------
  while(1)
  {
    // just loop and do nothing
  }
}
//-----------------------------------------------------------------------------





