//*****************************************************************************
//  SGBigNumbers.c
//  These are digits 0-9 and +-.'AvWCF and each digit is arranged as 4 custom text
//  characters drawn as a block of 2x2 text characters. Because these
//  big digits are drawn as text characters they draw VERY fast and can
//  be drawn with a simple function (that draws 4 text chars).
//  All digits need 32 bytes (to hold data for 4 custom text chars).
//  The bytes are arranged so they can be written direct to CG ram;
//  Each custom char is 8 bytes (down) then the big digit is 2x2 chars.
//  I have supplied 4 user functions; see below.
//  NOTE!! GCLD is set to 6x8 font mode (40x16 chars mode).
//*****************************************************************************

// SGBN functions (see page bottom);
void SGBN_Setup_CG(void);      // sets the GC RAM as block3 in GLCD RAM
void SGBN_Save_CG_Chars(void);  // saves the 10 digits as 40 custom text characters
void SGBN_Send_Char(unsigned char dig, unsigned char x, unsigned char y, unsigned char mode);     // internal use
void SGBN_Write_Char(unsigned char dig, unsigned char x, unsigned char y, unsigned char mode);    // writes a large digit to screen
void SGBN_Write_Text(unsigned char *btext, unsigned char x, unsigned char y, unsigned char mode); // writes a big digit text string 

// the 19 digits (0123456789+-.'AvWCF)
const unsigned char SGBNdigits[(19*4)*8] = {
// digit 0 (all digits are 32 bytes each)
0b00000000,0b00000111,0b00001111,0b00011100,0b00011000,0b00011000,0b00011000,0b00011000,
0b00000000,0b00111000,0b00111100,0b00001110,0b00000110,0b00001110,0b00011110,0b00110110,
0b00011001,0b00011011,0b00011110,0b00011100,0b00011100,0b00001111,0b00000111,0b00000000,
0b00100110,0b00000110,0b00000110,0b00000110,0b00001110,0b00111100,0b00111000,0b00000000,
// digit 1
0b00000000,0b00000001,0b00000011,0b00000111,0b00000111,0b00000001,0b00000001,0b00000001,
0b00000000,0b00100000,0b00100000,0b00100000,0b00100000,0b00100000,0b00100000,0b00100000,
0b00000001,0b00000001,0b00000001,0b00000001,0b00000001,0b00000111,0b00000111,0b00000000,
0b00100000,0b00100000,0b00100000,0b00100000,0b00100000,0b00111000,0b00111000,0b00000000,
// digit 2
0b00000000,0b00000111,0b00001111,0b00011100,0b00011000,0b00000000,0b00000000,0b00000000,
0b00000000,0b00111000,0b00111100,0b00001110,0b00000110,0b00000110,0b00001110,0b00011100,
0b00000000,0b00000001,0b00000011,0b00000111,0b00001110,0b00011111,0b00011111,0b00000000,
0b00111000,0b00110000,0b00100000,0b00000000,0b00000000,0b00111110,0b00111110,0b00000000,
// digit 3
0b00000000,0b00011111,0b00011111,0b00000000,0b00000000,0b00000001,0b00000001,0b00000000,
0b00000000,0b00111110,0b00111110,0b00011100,0b00111000,0b00110000,0b00110000,0b00111000,
0b00000000,0b00000000,0b00000000,0b00011000,0b00011100,0b00001111,0b00000111,0b00000000,
0b00011100,0b00001110,0b00000110,0b00000110,0b00001110,0b00111100,0b00111000,0b00000000,
// digit 4
0b00000000,0b00000000,0b00000000,0b00000001,0b00000011,0b00000111,0b00001110,0b00011100,
0b00000000,0b00011000,0b00111000,0b00111000,0b00111000,0b00011000,0b00011000,0b00011000,
0b00011000,0b00011111,0b00011111,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,
0b00011000,0b00111110,0b00111110,0b00011000,0b00011000,0b00011000,0b00011000,0b00000000,
// digit 5
0b00000000,0b00011111,0b00011111,0b00011000,0b00011000,0b00011111,0b00011111,0b00000000,
0b00000000,0b00111110,0b00111110,0b00000000,0b00000000,0b00111000,0b00111100,0b00001110,
0b00000000,0b00000000,0b00000000,0b00011000,0b00011100,0b00001111,0b00000111,0b00000000,
0b00000110,0b00000110,0b00000110,0b00000110,0b00001110,0b00111100,0b00111000,0b00000000,
// digit 6
0b00000000,0b00000001,0b00000011,0b00000111,0b00001110,0b00011100,0b00011000,0b00011111,
0b00000000,0b00111000,0b00111000,0b00000000,0b00000000,0b00000000,0b00000000,0b00111000,
0b00011111,0b00011100,0b00011000,0b00011000,0b00011100,0b00001111,0b00000111,0b00000000,
0b00111100,0b00001110,0b00000110,0b00000110,0b00001110,0b00111100,0b00111000,0b00000000,
// digit 7
0b00000000,0b00011111,0b00011111,0b00000000,0b00000000,0b00000000,0b00000000,0b00000001,
0b00000000,0b00111110,0b00111110,0b00000110,0b00001110,0b00011100,0b00111000,0b00110000,
0b00000011,0b00000111,0b00000110,0b00000110,0b00000110,0b00000110,0b00000110,0b00000000,
0b00100000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,
// digit 8
0b00000000,0b00000111,0b00001111,0b00011100,0b00011000,0b00011000,0b00011100,0b00001111,
0b00000000,0b00111000,0b00111100,0b00001110,0b00000110,0b00000110,0b00001110,0b00111100,
0b00001111,0b00011100,0b00011000,0b00011000,0b00011100,0b00001111,0b00000111,0b00000000,
0b00111100,0b00001110,0b00000110,0b00000110,0b00001110,0b00111100,0b00111000,0b00000000,
// digit 9
0b00000000,0b00000111,0b00001111,0b00011100,0b00011000,0b00011000,0b00011100,0b00001111,
0b00000000,0b00111000,0b00111100,0b00001110,0b00000110,0b00000110,0b00001110,0b00111110,
0b00000111,0b00000000,0b00000000,0b00000000,0b00000000,0b00000111,0b00000111,0b00000000,
0b00111110,0b00000110,0b00001110,0b00011100,0b00111000,0b00110000,0b00100000,0b00000000,
// digit +
0b00000000,0b00000000,0b00000000,0b00000001,0b00000001,0b00000001,0b00000001,0b00011111,
0b00000000,0b00000000,0b00000000,0b00100000,0b00100000,0b00100000,0b00100000,0b00111110,
0b00011111,0b00000001,0b00000001,0b00000001,0b00000001,0b00000000,0b00000000,0b00000000,
0b00111110,0b00100000,0b00100000,0b00100000,0b00100000,0b00000000,0b00000000,0b00000000,
// digit -
0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00011111,
0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00111110,
0b00011111,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,
0b00111110,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,
// digit .
0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,
0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,
0b00000000,0b00000000,0b00000000,0b00001111,0b00001111,0b00001111,0b00001111,0b00000000,
0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,
// digit '
0b00000000,0b00000111,0b00000111,0b00000001,0b00000011,0b00000111,0b00000110,0b00000000,
0b00000000,0b00100000,0b00100000,0b00100000,0b00100000,0b00000000,0b00000000,0b00000000,
0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,
0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,
// digit A
0b00000000,0b00000111,0b00001111,0b00011100,0b00011000,0b00011000,0b00011000,0b00011000,
0b00000000,0b00111000,0b00111100,0b00001110,0b00000110,0b00000110,0b00000110,0b00000110,
0b00011000,0b00011111,0b00011111,0b00011000,0b00011000,0b00011000,0b00011000,0b00000000,
0b00000110,0b00111110,0b00111110,0b00000110,0b00000110,0b00000110,0b00000110,0b00000000,
// digit v
0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00011000,0b00011000,0b00011000,
0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000110,0b00000110,0b00000110,
0b00011000,0b00011000,0b00011100,0b00001110,0b00000111,0b00000011,0b00000001,0b00000000,
0b00000110,0b00000110,0b00001110,0b00011100,0b00111000,0b00110000,0b00100000,0b00000000,
// digit W
0b00000000,0b00011000,0b00011000,0b00011000,0b00011000,0b00011000,0b00011000,0b00011001,
0b00000000,0b00000110,0b00000110,0b00000110,0b00000110,0b00000110,0b00000110,0b00100110,
0b00011001,0b00011001,0b00011001,0b00011001,0b00011111,0b00001111,0b00000110,0b00000000,
0b00100110,0b00100110,0b00100110,0b00100110,0b00111110,0b00111100,0b00011000,0b00000000,
// digit C
0b00000000,0b00000111,0b00001111,0b00011100,0b00011000,0b00011000,0b00011000,0b00011000,
0b00000000,0b00111000,0b00111100,0b00001110,0b00000110,0b00000000,0b00000000,0b00000000,
0b00011000,0b00011000,0b00011000,0b00011000,0b00011100,0b00001111,0b00000111,0b00000000,
0b00000000,0b00000000,0b00000000,0b00000110,0b00001110,0b00111100,0b00111000,0b00000000,
// digit F
0b00000000,0b00011111,0b00011111,0b00011000,0b00011000,0b00011000,0b00011000,0b00011111,
0b00000000,0b00111110,0b00111110,0b00000000,0b00000000,0b00000000,0b00000000,0b00111000,
0b00011111,0b00011000,0b00011000,0b00011000,0b00011000,0b00011000,0b00011000,0b00000000,
0b00111000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000,0b00000000
};

//-----------------------------------------------------------------------------
// functions for SGBigNumbers.c  below;
//-----------------------------------------------------------------------------

//=============================================================================
//  SGBN_SETUP_CG     
//=============================================================================
void SGBN_Setup_CG(void)    
{
  //-------------------------------------------------------
  // this must be called first before calling other SGBN functions.
  // All it does is tell the GLCD that there is CG ram now
  // assigned to GLCD ram block 3.
  //-------------------------------------------------------
  // Special GLCD setup for custom characters.
  // Do offset register set to set character ram address to the first
  // available block of 2048 ram, which is block 3 (at 6144)
  T6963C_writeData(0x03);     // ram block 3 (each block = 2048)
  T6963C_writeData(0x00);     // must be zero
  T6963C_writeCommand(0x22);  // command for "offset register set"
}
//-----------------------------------------------------------------------------


//=============================================================================
//  SGBN_SAVE_CG_CHARS     
//=============================================================================
void SGBN_Save_CG_Chars(void) 
{
  //-------------------------------------------------------
  // This writes the data for all 19 big digits (0123456789+-.'AvWCF)
  // As each big digit is 4 custom text chars, we write 13*4 *8
  // bytes in total.
  // The data is already formatted correctly in array SGBNdigits[]
  // so we can just write all the bytes sequentially.
  // The CG ram is on block 3 and the user CG ram starts at character
  // 128 so that is (3*2048)+(128*8) = 6144+1024 = 7168 = 0x1C00
  //-------------------------------------------------------
  unsigned int SGBN_i;

  // set the start address for the first byte we write to CG ram.
  T6963C_writeData(0x00);     // ram lo address
  T6963C_writeData(0x1C);     // ram hi address
  T6963C_writeCommand(0x24);  // command to set ram address pointer
  
  // loop and write one byte to GC ram at a time!
  for(SGBN_i=0; SGBN_i<(19*4*8); SGBN_i++)  
  {
    T6963C_writeData(SGBNdigits[SGBN_i]); // bits; xxBBBBBB
    T6963C_writeCommand(0xC0);    // command to write byte to graphic ram and inc ram pointer
  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//  SGBN_SEND_CHAR     
//=============================================================================
void SGBN_Send_Char(unsigned char d, unsigned char x, unsigned char y, unsigned char mode) 
{
  //-------------------------------------------------------
  // Not for user to call! This is called only by SGBN_Write_char()
  //-------------------------------------------------------
  d = (d * 4);    // convert to CG ram offset
  T6963C_Write_Char(128+32+0+d,x,y,mode);  // write 4 custom chars = 1 big digit 
  T6963C_Write_Char(128+32+1+d,x+1,y,mode);
  T6963C_Write_Char(128+32+2+d,x,y+1,mode);
  T6963C_Write_Char(128+32+3+d,x+1,y+1,mode);
}
//-----------------------------------------------------------------------------


//=============================================================================
//  SGBN_WRITE_CHAR     
//=============================================================================
void SGBN_Write_Char(unsigned char dig, unsigned char x, unsigned char y, unsigned char mode) 
{
  //-------------------------------------------------------
  // This writes one big digit to the screen.
  // Coords are XY 40x16 the same usage as T6963C_Write_Char()
  // Each big digit is written to the GLCD simply by writing 4 custom
  // text characters, in a 2x2 block.
  // Note! the CG ram starts at character 128, but MikroC functions
  // do an offset of -32 so to draw char 128 we tell MikroC to draw 128+32.
  //-------------------------------------------------------
  // do special cases;  +-.'AvWCF
  if(dig < '0' || dig > '9')   // if special, decode to a valid digit
  {
    if(dig==' ')  // special case for space, uses 4 normal space chars
    {
      T6963C_Write_Char(' ',x,y,mode);  // write 4 custom chars = 1 big digit 
      T6963C_Write_Char(' ',x+1,y,mode);
      T6963C_Write_Char(' ',x,y+1,mode);
      T6963C_Write_Char(' ',x+1,y+1,mode);
      return;
    }
    if(dig=='+')  dig = 10;     // these all use custom chars
    if(dig=='-')  dig = 11;
    if(dig=='.')  dig = 12;
    if(dig==39)   dig = 13;  // ' char
    if(dig=='A' || dig=='a') dig = 14;  
    if(dig=='V' || dig=='v') dig = 15;     
    if(dig=='W' || dig=='w') dig = 16;     
    if(dig=='C' || dig=='c') dig = 17;     
    if(dig=='F' || dig=='f') dig = 18;     
  }
  else    // else is digit 0-9
  {
    dig -= '0';       // convert ascii '0'-'9' to 0-9 integer
  }
  // now make the character
  SGBN_Send_Char(dig,x,y,mode);
}
//-----------------------------------------------------------------------------


//=============================================================================
//  SGBN_WRITE_TEXT    
//=============================================================================
void SGBN_Write_Text(unsigned char *btext, unsigned char x, unsigned char y, unsigned char mode) 
{
  //-------------------------------------------------------
  // This is just a wrap function that lets the user write
  // a string of big digits instead of just 1 big digit char.
  // See; SGBN_Write_Char()
  //-------------------------------------------------------
  unsigned char SGBN_d;     // digit pointer
  // now just print each big digit until NULL found
  SGBN_d = 0;
  while(btext[SGBN_d])
  {
    SGBN_Write_Char(btext[SGBN_d],x,y,mode); // write one digit
    x+=2;                           // move across 1 big digit
    if(x >= 40) break;              // break if hit edge of screen!
    SGBN_d++;                       // inc to next digit
  }
}
//-----------------------------------------------------------------------------




