/******************************************************************************
    SG_Serial_Term.c  A serial receive terminal, showing ascii or hex+ascii.
    Copyright:        Open-Source, Feb2011, written by www.RomanBlack.com
    MCU:              P18F8527
    Dev.Board:        SmartGLCD (SmartGLCD 240x128 module)
    Oscillator:       HSPLL 32.0 MHz (8MHz xtal)
    Compiler:         MikroC v7.0 (or MikroC PRO 4.0)

  This project is always receiving data from the serial port (via usart)
  and storing it in rdat[]. When rdat[] is full it either stops storing it
  or it loops and keeps storing from the top again (user selected).
  When it has time, the main() draws the received data to the terminal
  screen until all received bytes are drawn.
  There are 2 user screens; setup screen and terminal screen.
  Data will still be received even when the setup screen in shown.
  Terminal screen has 2 modes; 640 chars, shown as ascii on 40x16 text grid.
  Or 160 chars, shown as both hex and ascii. Only the first 160 chars of the
  total 640 will be shown, in a style like the old dos terminals.
  There are some custom text chars used to show things like line feed chars
  etc in the ascii data, this makes it more useful for displaying received
  serial data.
  NOTE! There is now storage of up to 2560 received bytes, shown as 4 screens!
******************************************************************************/
// Global Variables
unsigned char tp_x;     // will hold results of Touch Panel adc values
unsigned char tp_y;

unsigned char rdat[640*4];  // holds 40x16 received chars *4 screens max
unsigned int rdat_address;  // pointer for received bytes memory
unsigned int rdat_addlast;  // last byte that has been displayed
unsigned char screen;       // which screen to show
unsigned int screen_lowbyte;
unsigned int screen_highbyte;
unsigned int i;

unsigned char txt[6];  // holds number for text display

unsigned char mode_setup;         // 1=setup screen, 0=terminal screen
unsigned char mode_baudrate;      // the receive baudrate

unsigned char check_clear;        // used for selected modes
unsigned char check_showhex;

#include "T6963C_MikroC.h"      // include this if using MikroC (older)
//#include "T6963C_MikroC_PRO.h"  // include this if using MikroC PRO

#include "TPfast.c"     // needed for my TouchPanel functions 

// my functions used in this tutorial
void Make_Custom_Characters(void);
void Draw_Setup_Screen(unsigned char draw_all);
void Draw_Terminal_Screen(void);
void Change_Baudrate(void);
unsigned char Get_Ascii_Char(unsigned char);
void ByteToHex(unsigned char btin, unsigned char *strout);
void Draw_Terminal_Char(unsigned int);

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void interrupt()
{
  //-------------------------------------------------------
  // TMR0 overflow int; prescale 1:1, gets here every 256 instructions.
  //-------------------------------------------------------
  if(Usart_Data_Ready() && rdat_address < (640*4))
  {
    rdat[rdat_address] = Usart_Read();
    rdat_address++;
  }
  
  INTCON.TMR0IF = 0;    // clear the int flag before exit
}
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


//=============================================================================
//  BLANK
//=============================================================================
void blank(void)
{
  //-------------------------------------------------------

}
//-----------------------------------------------------------------------------

//=============================================================================
//  MAIN
//=============================================================================
void main()
{
  //-------------------------------------------------------
  // setup PIC 18F8527 for SmartGLCD pins
  CMCON = 0x07;        // turn off comparators (make all pins digital)
  ADCON0 = 0b00000001;  // ADC module on
  ADCON1 = 0b00001101;  // AN0,AN1 are adc inputs, 0v-5v range
  ADCON2 = 0b00110010;  // ADC result left justified (0-255 range)

  LATA =  0b00000000;
  TRISA = 0b00000011;   // RA0,RA1 analog inputs (TP)
  LATC =  0b00000110;   // LEDs off at start
  TRISC = 0b00000000;   // C1, C2 backlight LED
  LATG =  0b00000001;   // LED off at start
  TRISG = 0b00000000;   // G0 backlight LED
  
  LATJ  = 0b01000000;   // RJ6=FS (1=font6 0=font8), RJ5=MD
  TRISJ = 0b00000000;   // GLCD control port

  BacklightRed    = 1;     // control the GLCD backlight leds; 0=on, 1=off
  BacklightGreen  = 0;     // green ON
  BacklightBlue   = 1;

  T0CON = 0b11000000;   // TMR0 enabled, 8bit, 1:1 prescaler

  //-------------------------------------------------------
  // startup delay, let the PSU voltage stabilise etc.
  Delay_ms(30);

  // Initialize T6963C GLCD
  //T6963C_init(240, 128, 6);   // init for MikroC PRO version
  T6963C_init(240, 128, 6, &PORTH, &PORTJ, 2, 1, 0, 4); // init for MikroC version
  T6963C_graphics(1);       // graphics mode = on
  T6963C_text(1);           // text mode = on (now both are on)
  T6963C_cursor(0);         // cursor = off

  //-------------------------------------------------------
  // initialise stuff
  // setup USART to receive serial
  mode_baudrate = 2;    // starting default is 2 = 9600 baud
  Change_Baudrate();

  Make_Custom_Characters(); // custom text characters used in display
  mode_setup = 1;           // start with setup screen
  check_showhex = 0;
  check_clear = 0;
  screen = 0;               // default is first screen
  Draw_Setup_Screen(1);     // 1=draw ALL

  for(i=0; i<(640*4); i++) rdat[i] = 0; // clear received data in memory
  rdat_address = 0;                 // and its pointers
  rdat_addlast = 0;

          // TEMP! load some dummy data in for testing
          memcpy(rdat,"This is a test...",18);
          rdat[18] = 13;
          rdat[19] = 10;
          rdat[20] = 3;
          rdat[21] = 140;
          rdat_address = 22;   // 22 bytes loaded
          

  // finally turn TMR0 interrupt on (used for usart receive)
  INTCON.GIE = 1;
  INTCON.TMR0IE = 1;

  //-------------------------------------------------------
  // main running loop; This has 2 modes, in setup mode it checks
  // the buttons and changes the settings on the setup screem.
  // Data can still be received when in setup screen mode.
  // In the other mode it is just the terminal screen and all it does
  // is draw the serial data to screen.
  //-------------------------------------------------------
  while(1)
  {
    Delay_mS(30);   // small delay for fast button press response
    //---------------------------------------------
    // setup screen mode! Here it gets touchpanel XY and tests
    // for the buttons pressed and changes settings etc.
    //---------------------------------------------
    if(mode_setup)
    {
      // get touchpanel XY 
      TPfast_SetX();          // read X 
      Delay_mS(2);
      TPfast_StartRead();   
      tp_x = TPfast_Read();
      TPfast_SetY();          // read Y
      Delay_mS(2);
      TPfast_StartRead();  
      tp_y = TPfast_Read();
              // (TEMP) format and display TP X value to screen
              ByteToStr(tp_x,txt+1);
              txt[0] = 'x';
              T6963C_Write_Text(txt, 23, 12, T6963C_ROM_MODE_XOR);
              // format and display TP Y value to screen
              ByteToStr(tp_y,txt);
              txt[0] = 'y';
              T6963C_Write_Text(txt, 32, 12, T6963C_ROM_MODE_XOR);
      
      // process the buttons.
      // select screen button
      if(tp_y > 171 && tp_y < 197 && tp_x > 42 && tp_x < 212)    
      {
        screen++;     // select new terminal screen 0-3
        if(screen >= 4) screen = 0;
        screen_lowbyte = (screen * 640);          // get address of lowest byte shown on screen
        screen_highbyte = (screen_lowbyte + 639); // get address of highest byte shown on screen
        Draw_Setup_Screen(0);     // 0=draw buttons only
        Delay_mS(350);
      }
      // if showhex button
      if(tp_y > 133 && tp_y < 158 && tp_x > 42 && tp_x < 212)    
      {
        if(check_showhex) check_showhex = 0;  // toggle mode
        else              check_showhex = 1;
        Draw_Setup_Screen(0);     // 0=draw buttons only
        Delay_mS(300);
      }
      // if baudrate buttons
      if(tp_y > 91 && tp_y < 119)    // if either baudrate button
      {
        if(tp_x > 40 && tp_x < 70)    // if baudrate down button
        {
          if(mode_baudrate) mode_baudrate--;  // change down
          Change_Baudrate();
          Draw_Setup_Screen(0);     // 0=draw buttons only
          Delay_mS(300);
        }
        if(tp_x > 186 && tp_x < 216)    // if baudrate up button
        {
          if(mode_baudrate < 5) mode_baudrate++;  // change up
          Change_Baudrate();
          Draw_Setup_Screen(0);     // 0=draw buttons only
          Delay_mS(400);
        }
      }
      // bottom buttons
      if(tp_y > 26 && tp_y < 58)    // if either bottom button
      {
        if(tp_x > 17 && tp_x < 108)    // if clear memory button
        {
          for(i=0; i<(640*4); i++) rdat[i] = 0; // clear all received data in memory
          rdat_address = 0;                     // and its pointers
          rdat_addlast = 0;
          check_clear = 1;      // draw black button to show it cleared
          Draw_Setup_Screen(0); // 0=draw buttons only
          check_clear = 0;
          Change_Baudrate();    // refresh usart ready to receive again
        }
        if(tp_x > 131 && tp_x < 232)    // if terminal screen button
        {
          T6963C_panelFill(0);    // clear screen
          mode_setup = 0;         // change to display terminal mode
          while(tp_y > 22)        // wait until panel is NOT pressed
          {
            TPfast_StartRead();  
            tp_y = TPfast_Read();
          }
          Delay_mS(100);
          Draw_Terminal_Screen(); // is now in terminal screen mode
          continue;
        }
      }
    }
    //---------------------------------------------
    // Terminal screen mode! Here it just draws any new received chars
    // to the terminal screen. There are 4 terminal screens of 640 bytes
    // each giving a total data storage and display of 2560 bytes.
    // Terminal mode is kept fast so it draws data to screen fast,
    // so it only checks the touchpanel Y and exits back to setup mode
    // if ANY screen location is touched.
    //---------------------------------------------
    else    // else is terminal screen mode
    {
      // draw any incoming chars to the screen as they arrive
      while(rdat_address > rdat_addlast)  // if any bytes were received
      {
        if(rdat_addlast >= screen_lowbyte && rdat_addlast <= screen_highbyte)        
        {
          Draw_Terminal_Char(rdat_addlast); // (if onscreen) draw the byte to screen
        }
        rdat_addlast++;                     // inc the count
      }
      
      // check for y pressed, if so, go back to setup mode
      TPfast_StartRead();  
      tp_y = TPfast_Read();
      if(tp_y > 22)     // if panel touched anywhere
      {
        mode_setup = 1;
        T6963C_panelFill(0);    // clear screen
        while(tp_y > 22)        // wait until panel is NOT pressed
        {
          TPfast_StartRead();  
          tp_y = TPfast_Read();
        }
        Draw_Setup_Screen(1);     // 1=draw all
      }
    }
    //---------------------------------------------
  } 
}
//-----------------------------------------------------------------------------

                                       
//=============================================================================
//  DRAW_SETUP_SCREEN
//=============================================================================
void Draw_Setup_Screen(unsigned char draw_all)
{
  //-------------------------------------------------------
  // this draws ALL the setup screen (like when screen is changed),
  // or just the text buttons (like when a button is pressed)
  //-------------------------------------------------------
  #define BUT_LEFT    32
  #define BUT_RIGHT   (240-BUT_LEFT)
  #define BUT_HEIGHT  14

  if(draw_all)
  {
    // draw any text that does not change
    T6963C_Write_Text("Serial Receive Terminal Setup", 5, 0, T6963C_ROM_MODE_XOR);
    T6963C_Write_Text("Terminal screen ",11, 3, T6963C_ROM_MODE_XOR);
    T6963C_Write_Text("Show data as Hex",     12, 6, T6963C_ROM_MODE_XOR);
    T6963C_Write_Text("Clear memory",         3, 14, T6963C_ROM_MODE_XOR);
    T6963C_Write_Text("Terminal screen",      22, 14, T6963C_ROM_MODE_XOR);

    // draw the boxes around the buttons
    T6963C_rectangle(BUT_LEFT, 20, BUT_RIGHT, 20+BUT_HEIGHT, T6963C_WHITE);  
    T6963C_rectangle(BUT_LEFT, 44, BUT_RIGHT, 44+BUT_HEIGHT, T6963C_WHITE);  
    T6963C_rectangle(BUT_LEFT, 68, BUT_RIGHT, 68+BUT_HEIGHT, T6963C_WHITE);  
    T6963C_rectangle(9, 108, 100, 108+BUT_HEIGHT, T6963C_WHITE);  
    T6963C_rectangle(124, 108, 230, 108+BUT_HEIGHT, T6963C_WHITE);  

    // draw the checkbox
    T6963C_rectangle(BUT_LEFT+3, 44+3, BUT_LEFT+3+8, 44+3+8, T6963C_WHITE);  

    // special stuff for baudrate select box
    T6963C_Line(BUT_LEFT+25,68,BUT_LEFT+25,68+BUT_HEIGHT,T6963C_WHITE);
    T6963C_Line(BUT_RIGHT-25,68,BUT_RIGHT-25,68+BUT_HEIGHT,T6963C_WHITE);
    T6963C_Write_Char(4,7, 9, T6963C_ROM_MODE_XOR);    // down arrow
    T6963C_Write_Char(3,32, 9, T6963C_ROM_MODE_XOR);   // up arrow
  }

  // now draw the changing stuff.
  // the two check boxes;
  if(check_showhex)     T6963C_box(BUT_LEFT+4, 44+4, BUT_LEFT+4+7, 44+4+6, T6963C_WHITE); 
  else                  T6963C_box(BUT_LEFT+4, 44+4, BUT_LEFT+4+7, 44+4+6, T6963C_BLACK); 

  if(check_clear)
  {
    T6963C_box(9+1, 108+1, 100, 108+BUT_HEIGHT-1, T6963C_WHITE);
    Delay_mS(200);  
    T6963C_box(9+1, 108+1, 100, 108+BUT_HEIGHT-1, T6963C_BLACK);
  }
  
  if(mode_baudrate == 0)  T6963C_Write_Text("  2400 Baud",14, 9, T6963C_ROM_MODE_XOR);
  if(mode_baudrate == 1)  T6963C_Write_Text("  4800 Baud",14, 9, T6963C_ROM_MODE_XOR);
  if(mode_baudrate == 2)  T6963C_Write_Text("  9600 Baud",14, 9, T6963C_ROM_MODE_XOR);
  if(mode_baudrate == 3)  T6963C_Write_Text(" 19200 Baud",14, 9, T6963C_ROM_MODE_XOR);
  if(mode_baudrate == 4)  T6963C_Write_Text(" 57600 Baud",14, 9, T6963C_ROM_MODE_XOR);
  if(mode_baudrate >= 5)  T6963C_Write_Text("115200 Baud",14, 9, T6963C_ROM_MODE_XOR);

  if(screen == 0)  T6963C_Write_Char('0',27, 3, T6963C_ROM_MODE_XOR);
  if(screen == 1)  T6963C_Write_Char('1',27, 3, T6963C_ROM_MODE_XOR);
  if(screen == 2)  T6963C_Write_Char('2',27, 3, T6963C_ROM_MODE_XOR);
  if(screen == 3)  T6963C_Write_Char('3',27, 3, T6963C_ROM_MODE_XOR);

        // (temp) display custom text chars
        T6963C_Write_Char(0, 6, 12, T6963C_ROM_MODE_XOR);    // down arrow
        T6963C_Write_Char(1, 7, 12, T6963C_ROM_MODE_XOR);    // down arrow
        T6963C_Write_Char(2, 8, 12, T6963C_ROM_MODE_XOR);    // down arrow
        T6963C_Write_Char(3, 9, 12, T6963C_ROM_MODE_XOR);    // down arrow
        T6963C_Write_Char(4, 10, 12, T6963C_ROM_MODE_XOR);    // down arrow
        T6963C_Write_Char(5, 11, 12, T6963C_ROM_MODE_XOR);    // down arrow
        T6963C_Write_Char(6, 12, 12, T6963C_ROM_MODE_XOR);    // down arrow
}
//-----------------------------------------------------------------------------


//=============================================================================
//  MAKE_CUSTOM_CHARACTERS
//=============================================================================
void Make_Custom_Characters(void)
{
  //-------------------------------------------------------
  // most of this copied from my Tutorial 09; SG_Adv_Text.c
  //-------------------------------------------------------
  // Special GLCD setup for custom characters.
  // Do offset register set to set character ram address to the first
  // available block of 2048 ram, which is block 3 (at 6144)
  T6963C_writeData(0x03);     // ram block 3 (each block = 2048)
  T6963C_writeData(0x00);     // must be zero
  T6963C_writeCommand(0x22);  // command for "offset register set"

  // Set the ram address for the CG character at location 224 which
  // will be displayed my MikroC char functions at location 0 as MikroC
  // references text as -32. (so 0-32 becomes 224)
  // The ram address is (block*3)+(224*8) because there are 8 bytes per character.
  // So ram address is 6144+1792 = 7936 = 0x1F00
  T6963C_writeData(0x00);     // ram lo address
  T6963C_writeData(0x1F);     // ram hi address
  T6963C_writeCommand(0x24);  // command to set ram address pointer
  
  // Now write the 8 bytes to draw a custom character.
  // format is horizontal bytes, top down. As we are in 6x8 font mode
  // only the right hand 6 bits are used, and the bit5 should be blank
  // as there should (normally) be 5 character bits and one blank space bit.
  // With text characters the bottom byte is normally blank too.
  
  // NOTE! as address is auto incrementing we can just keep adding more chars.
  // make custom character; ZERO BLOCK
  T6963C_writeData(0b00011111); // bits; xxBBBBBB
  T6963C_writeCommand(0xC0);    // command to write byte to graphic ram and inc ram pointer
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00000000);     
  T6963C_writeCommand(0xC0);  

  // make custom character; LOW BLOCK (represesnts low bad chars)
  T6963C_writeData(0b00011111); // bits; xxBBBBBB
  T6963C_writeCommand(0xC0);    // command to write byte to graphic ram and inc ram pointer
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00010001);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00000000);     
  T6963C_writeCommand(0xC0);  

  // make custom character; HIGH BLOCK (represesnts high bad chars)
  T6963C_writeData(0b00011111); // bits; xxBBBBBB
  T6963C_writeCommand(0xC0);    // command to write byte to graphic ram and inc ram pointer
  T6963C_writeData(0b00010001);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00000000);     
  T6963C_writeCommand(0xC0);  

  // make custom character; UP ARROW
  T6963C_writeData(0b00000100); // bits; xxBBBBBB
  T6963C_writeCommand(0xC0);    // command to write byte to graphic ram and inc ram pointer
  T6963C_writeData(0b00000100);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00001110);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00001110);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00000000);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00000000);     
  T6963C_writeCommand(0xC0);  

  // make custom character; DOWN ARROW
  T6963C_writeData(0b00000000); // bits; xxBBBBBB    
  T6963C_writeCommand(0xC0);    // command to write byte to graphic ram and inc ram pointer
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00001110);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00001110);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00000100);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00000100);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00000000);     
  T6963C_writeCommand(0xC0);  

  // 2 special chars for the 13,10 line feed pair
  // make custom character; 13
  T6963C_writeData(0b00011111); // bits; xxBBBBBB
  T6963C_writeCommand(0xC0);    // command to write byte to graphic ram and inc ram pointer
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011011);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011011);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00010011);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00000000);     
  T6963C_writeCommand(0xC0);  

  // make custom character; 10
  T6963C_writeData(0b00011111); // bits; xxBBBBBB
  T6963C_writeCommand(0xC0);    // command to write byte to graphic ram and inc ram pointer
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00010001);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00010101);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00010001);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00011111);     
  T6963C_writeCommand(0xC0);  
  T6963C_writeData(0b00000000);     
  T6963C_writeCommand(0xC0);  
}
//-----------------------------------------------------------------------------


//=============================================================================
//  CHANGE_BAUDRATE
//=============================================================================
void Change_Baudrate(void)
{
  //-------------------------------------------------------
  // changes the USART baudrate and also draws the
  // new baudrate text to the screen.
  if(mode_baudrate == 0)  Usart_Init(2400);
  if(mode_baudrate == 1)  Usart_Init(4800);
  if(mode_baudrate == 2)  Usart_Init(9600);
  if(mode_baudrate == 3)  Usart_Init(19200);
  if(mode_baudrate == 4)  Usart_Init(57600);
  if(mode_baudrate >= 5)  Usart_Init(115200);
}
//-----------------------------------------------------------------------------


//=============================================================================
//  GET_ASCII_CHAR
//=============================================================================
unsigned char Get_Ascii_Char(unsigned char btin)
{
  //-------------------------------------------------------
  // this allows display of custom characters instead of
  // non-visible ascii characters. This lets the user see things
  // like line feed chars in text etc.
  //-------------------------------------------------------
  // return with ascii if it is legal ascii char.
  if(btin==0 || (btin >= ' ' && btin <= 127)) return(btin);   // null and ascii 

  // now test for special cases, and return with a custom char to display.
  if(btin == 13) return(5);   // custom chars for line feed pair
  if(btin == 10) return(6);
  
  if(btin < ' ') return(1);   // custom char for low illegal char
  return(2);                  // finally custom char for high illegal char
}
//-----------------------------------------------------------------------------


//=============================================================================
//  BYTETOHEX
//=============================================================================
void ByteToHex(unsigned char btin, unsigned char *strout)
{
  //-------------------------------------------------------
  // converts a byte 0-255 to a hex string consisting of
  // 2 hex digits followed by a null. I don't think MikroC
  // has a conversion function like this?
  //-------------------------------------------------------
  unsigned char d1, d0;
  // convert the byte to 2 digits each 0-15
  d1 = (btin >> 4);
  d0 = (btin & 0x0F);
  // then put the 2 hex chars in string
  if(d1 <= 9) strout[0] = (d1 + '0');
  else
  {
    if(d1 == 10) strout[0] = 'a';
    if(d1 == 11) strout[0] = 'b';
    if(d1 == 12) strout[0] = 'c';
    if(d1 == 13) strout[0] = 'd';
    if(d1 == 14) strout[0] = 'e';
    if(d1 == 15) strout[0] = 'f';
  }
  if(d0 <= 9) strout[1] = (d0 + '0');
  else
  {
    if(d0 == 10) strout[1] = 'a';
    if(d0 == 11) strout[1] = 'b';
    if(d0 == 12) strout[1] = 'c';
    if(d0 == 13) strout[1] = 'd';
    if(d0 == 14) strout[1] = 'e';
    if(d0 == 15) strout[1] = 'f';
  }
  strout[2] = 0;    // add final null
}
//-----------------------------------------------------------------------------


//=============================================================================
//  DRAW_TERMINAL_CHAR
//=============================================================================
void Draw_Terminal_Char(unsigned int tcadd)
{
  //-------------------------------------------------------
  // draws one char to the serial receive terminal
  // but only if that char is in valid area. The var tcadd
  // is the address of the byte in rdat[] and covers 4 pages of 640 chars.
  // This handles both terminal screen modes and draws either
  // one ascii char or if in hex mode it draws a hex char pair and
  // an ascii char.
  //-------------------------------------------------------
  unsigned int tmath;
  unsigned char ch, tx, ty, trow;

  // derive tx and ty text coords from 40 modulus (fast system) 
  // as GLCD mode is 40x16 chars on screen
  tmath = tcadd;
  while(tmath >= 640) tmath -=640;  // remove screen offset!
  trow = 40;          // default 40 chars on a row
  if(check_showhex)   // if hex mode on
  {
    if(tmath >= 160) return;   // only show 160 chars in hex mode
    trow = 10;      // and 10 chars on a row
  }
  ty = 0;                
  while(tmath >= trow)  // fast modulus code to derive X and Y              
  {
    tmath -= trow;
    ty++;
  }
  tx = tmath;
    
  // get an ascii char to draw
  ch = Get_Ascii_Char(rdat[tcadd]); 

  // draw the ascii character if it is valid
  if(tcadd < rdat_address)   // if valid
  {
    if(!check_showhex)    // if 40x16 ascii mode
    {
      // draw the ascii char
      T6963C_Write_Char(ch,tx, ty, T6963C_ROM_MODE_XOR);   
    }
    else        // else show as hex/ascii, up to first 160 chars only
    {
      // draw the ascii char
      T6963C_Write_Char(ch,tx+30, ty, T6963C_ROM_MODE_XOR);   
      // draw the hex char
      ByteToHex(rdat[tcadd],txt);
      T6963C_Write_Text(txt,(tx*3), ty, T6963C_ROM_MODE_XOR);   
    }
  }  
}
//-----------------------------------------------------------------------------


//=============================================================================
//  DRAW_TERMINAL_SCREEN
//=============================================================================
void Draw_Terminal_Screen(void)
{
  //-------------------------------------------------------
  // this draws the received serial bytes either as ascii text
  // or as hex and ascii. Selected by check_showhex.
  // there are 4 screens of received data, each screen is 640
  // bytes.
  //-------------------------------------------------------
  unsigned int screen_offset;

  // loop and draw all valid chars that are on the current screen
  screen_offset = (screen * 640);
  for(i=screen_offset; i<screen_offset+640; i++)
  {
    Draw_Terminal_Char(i);  
  }
}
//-----------------------------------------------------------------------------









