/******************************************************************************
  BaudrateConvert.c   Dual USART bidirectional buffered baudrate converter!
  open-source  -  23rd Dec 2009  -  www.RomanBlack.com
  PIC 18F8520, 40MHz PLL, GLCD, BIGPIC6 dev board.

  This is a more complex application built with my multi-thread
  "PIC-thread loop frequency system". All tasks are performed in
  independent threads. For more info see; www.RomanBlack.com/PICthread.htm

  This project allows full asynchronous bidirectional exchange between
  2 USARTS at any baudrate, provided there are some pauses (so the buffer
  does not overrun) and that USART1 is the higher speed USART. The buffer
  usage is displayed on the GLCD as a 62-bar bargraph along with other info.
******************************************************************************/

// thread engine variables
unsigned char loopnum;
unsigned char intcount;

// thread dedicated variables
unsigned char th3timer;
unsigned char th4timer;

unsigned int th3display;
unsigned char th3bar;
unsigned char th3graph;
unsigned char th3x;


// buffer vars
#define MAXBUFFER1 1024
unsigned char buffer1[MAXBUFFER1];
unsigned int inpointer1;
unsigned int outpointer1;
unsigned int bufferload1;
unsigned char i;

// general vars
unsigned char buttons;
unsigned char uspeed1;
unsigned char uspeed2;

// functions
void setup_usarts(void);
void draw_main_screen(void);
void thread3body(void);
void thread4body(void);

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void interrupt()
{
  //----------------------------------------------------
  // this is the timer 2 int, set for 800 insts (80uS)
  // the main task is to update all the thread timers
  // every 10mS. 10mS = (80uS * 125).
	//----------------------------------------------------
  intcount++;
  if(intcount >= 125)     // if reached 10mS
  {
    intcount = 0;
    th3timer++;           // inc all the dedicated thread timers
    th4timer++;
  }
  PIR1.TMR2IF = 0;    // clear int flag and exit
}
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


//=============================================================================
//   MAIN
//=============================================================================
void main()
{
	//----------------------------------------------------
  // specific PIC18F8520 setup
  CMCON |= 0x07;              // turn off comparators
  ADCON1 |= 0x0D;             // Set AN0 and AN1 channel pins as analog
  LATB = 0;
  TRISB = 0b00001111;         // 4 buttons
  TRISC = 0b10000010;         // RC7=1, RC6=0 (usart1)
  TRISG = 0b00000100;         // RG2=1, RG1=0 (usart2)
  MEMCON.EBDIS = 1;           // disable external memory bus

	//----------------------------------------------------
  // setup vars
  inpointer1 = 0;
  outpointer1 = 0;
  uspeed1 = 3;      // starting baudrates
  uspeed2 = 1;
  setup_usarts();
  
	//----------------------------------------------------
  // GLCD setup
  Glcd_Init(&PORTJ, 0,1,2,3,5,4, &PORTD);   // pin setup on BIGPIC6
  Glcd_Fill(0x00);
  Glcd_Set_Font(Character8x8, 8, 8, 32);  
  draw_main_screen();   // draw to GLCD

  // TMR2 setup, used to make interrupt
  T2CON = 0b00000101;   // post 1:1, TMR2 on, pre 1:4 (1:4 total)
  PR2 = (200-1);        // 200*4 = 800 insts, (80uS)

  INTCON = 0b11000000;    // GIE on, PEIE on
  PIE1.TMR2IE = 1;        // TMR2 interrupt on
  
	//----------------------------------------------------
  // main multi-thread engine loop
  while(1)
  {
    // LOOP SYNCHRONISE ===================================
    // don't sync, just run loop at full speed
    loopnum++;

    // THREAD 0 ===========================================
    // HIGH PRIORITY!
    // check if usart1 received a byte, put it in buffer
    if(PIR1.RC1IF)
    {
      buffer1[inpointer1] = RCREG1;   // store the byte
      inpointer1++;
      if(inpointer1 >= MAXBUFFER1) inpointer1 = 0;
    }

    // THREAD 1 ===========================================
    // HIGH PRIORITY!
    // check if usart2 received a byte, send it out to usart1
    if(PIR3.RC2IF)
    {
      if(TXSTA1.TRMT) TXREG1 = RCREG2;  // if usart TX free, send byte
    }

    // THREAD 2 ===========================================
    // HIGH PRIORITY!
    // send usart1 byte if any bytes are in buffer
    bufferload1 = (inpointer1 - outpointer1); 
    if(bufferload1)
    {
      if(bufferload1 >= MAXBUFFER1) bufferload1 += MAXBUFFER1; // fix buffer roll
      if(th3display < bufferload1) th3display = bufferload1;   // record max load
      
      if(TXSTA2.TRMT)   // if usart free, send another byte
      {
        TXREG2 = buffer1[outpointer1];
        outpointer1++;
        if(outpointer1 >= MAXBUFFER1) outpointer1 = 0;  // fix buffer roll
      }
    }
    else  th3display = 0;
    
    // THREAD 3 ===========================================
    // timed interval
    // display buffer1 max load as a 62 bar bargraph on PORTB 
    if((loopnum & 0x3F) == 0)   // draw one bar every 64th loop
    {
      if(th3timer >= 10)   // update bargraph every 10*10mS (100mS)
      {
        thread3body();
      }
    }

    // THREAD 4 ===========================================
    // timed interval
    // test buttons, change the usart speeds and display on GLCD
    if(th4timer >= 30)   // do this thread every 30*10mS (300mS)
    {
      th4timer = 0;
      thread4body();
    }
    
    // THREAD 5 ===========================================
    // HIGH PRIORITY!
    // display USART errors
    if(RCSTA1 & 0b00000110)  
    {
      GLCD_Write_Text("ERROR1",(0*8),7,1);  
    }
    if(RCSTA2 & 0b00000110)  
    {
      GLCD_Write_Text("ERROR2",(8*8),7,1);  
    }
 
    // THREAD END =========================================
  }
}
//-----------------------------------------------------------------------------



//=============================================================================
//   THREAD 3 BODY
//=============================================================================
void thread3body(void)
{
  //-------------------------------------------------------
  // if this is the first bar then setup vars
  if(!th3bar) 
  { 
    th3graph = (th3display >> 4);   // convert 1024 to 64
    if(th3display && !th3graph) th3graph = 1;   // always min of 1 bar
    th3bar = 62;
    th3x = 125;
    return;
  }
  
  // now display one bar each time the thread is executed
  if(th3graph >= th3bar)   // draw a lit bar
  {
    GLCD_Dot(th3x,(3*8)+4,1);     // X,Y,col
  }
  else                     // else draw a empty bar
  {
    GLCD_Dot(th3x,(3*8)+4,0);     // X,Y,col
  }

  // and update chart and coords
  th3bar--;
  th3x -= 2;
  if(th3bar == 0) th3timer = 0;   // done!

}
//-----------------------------------------------------------------------------


//=============================================================================
//   THREAD 4 BODY
//=============================================================================
void thread4body(void)
{
  //-------------------------------------------------------
  // test buttons, change the usart speeds and display on GLCD
  // it doesn't matter here if we miss an incoming byte
  // because the user won't be changing baudrate in the middle
  // of sensitive data!
  // So just trash the buffers and reset everything back to scratch.
  //-------------------------------------------------------
  #define USPEED1LAST 7   
  #define USPEED2LAST 7

  buttons = (PORTB & 0x0F);     // test the 4 buttons
  if(buttons)           // if any buttons are pressed
  {
    if(buttons.F0)     // USART1 up pressed
    {
      if(uspeed1 < USPEED1LAST) uspeed1++;
    }
    if(buttons.F1)     // USART1 down pressed
    {
      if(uspeed1 > 0 && uspeed1 > uspeed2) uspeed1--;
    }
    if(buttons.F2)     // USART2 up pressed
    {
      if(uspeed2 < USPEED2LAST && uspeed2 < uspeed1) uspeed2++;
    }
    if(buttons.F3)     // USART2 down pressed
    {
      if(uspeed2 > 0) uspeed2--;
    }  

    // setup the usarts
    setup_usarts();

    // and draw it again to the main screen
    draw_main_screen();
  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//   DRAW MAIN SCREEN
//=============================================================================
void draw_main_screen(void)
{
  //-------------------------------------------------------
  // draw items that are always on screen
  Glcd_Set_Font(Character8x8, 8, 8, 32);     
  GLCD_Write_Text("Serial Convert",(0*8),0,1);  // title

  // draw the USART speeds
  GLCD_Write_Text("USART1:",(0*8),2,1);  
  if(uspeed1 == 0) GLCD_Write_Text("3200  ",(9*8),2,1);  
  if(uspeed1 == 1) GLCD_Write_Text("4800  ",(9*8),2,1);  
  if(uspeed1 == 2) GLCD_Write_Text("9600  ",(9*8),2,1);  
  if(uspeed1 == 3) GLCD_Write_Text("19200 ",(9*8),2,1);  
  if(uspeed1 == 4) GLCD_Write_Text("38400 ",(9*8),2,1);  
  if(uspeed1 == 5) GLCD_Write_Text("57600 ",(9*8),2,1);  
  if(uspeed1 == 6) GLCD_Write_Text("96000 ",(9*8),2,1);  
  if(uspeed1 == 7) GLCD_Write_Text("115200",(9*8),2,1);  

  GLCD_Write_Text("USART2:",(0*8),6,1);  
  if(uspeed2 == 0) GLCD_Write_Text("3200  ",(9*8),6,1);  
  if(uspeed2 == 1) GLCD_Write_Text("4800  ",(9*8),6,1);  
  if(uspeed2 == 2) GLCD_Write_Text("9600  ",(9*8),6,1);  
  if(uspeed2 == 3) GLCD_Write_Text("19200 ",(9*8),6,1);  
  if(uspeed2 == 4) GLCD_Write_Text("38400 ",(9*8),6,1);  
  if(uspeed2 == 5) GLCD_Write_Text("57600 ",(9*8),6,1);  
  if(uspeed2 == 6) GLCD_Write_Text("96000 ",(9*8),6,1);  
  if(uspeed2 == 7) GLCD_Write_Text("115200",(9*8),6,1);  

  //-------------------------------------------------------
  // draw the bargraph outer box
  GLCD_Rectangle(0,(3*8)+1,127,(3*8)+7,1);  // XYXYcol
  // and its label
  Glcd_Set_Font(System3x6, 3, 6, 32);     
  GLCD_Write_Text("USART1 IN BUFFER 1024",(2*8)+4,4,1);  

   // set big font again before exit 
  Glcd_Set_Font(Character8x8, 8, 8, 32);     
}
//-----------------------------------------------------------------------------




//=============================================================================
//   SETUP USARTS
//=============================================================================
void setup_usarts(void)
{
  //-------------------------------------------------------
  //BRG=0     40000000/64/baud -1 =  625000/baud -1
  //BRG=1     40000000/16/baud -1 = 2500000/baud -1
  if(uspeed1 < 3)
  {
    if(uspeed1 == 0) SPBRG1 = 194; // "3200"  
    if(uspeed1 == 1) SPBRG1 = 129; // "4800"  
    if(uspeed1 == 2) SPBRG1 = 64;  // "9600"  
    TXSTA1 = 0b00100000;    // BRGH = 0
  }
  else
  {
    if(uspeed1 == 3) SPBRG1 = 129; // "19200"  
    if(uspeed1 == 4) SPBRG1 = 64;  // "38400"  
    if(uspeed1 == 5) SPBRG1 = 42;  // "57600"  
    if(uspeed1 == 6) SPBRG1 = 25;  // "96000"  
    if(uspeed1 == 7) SPBRG1 = 21;  // "115200"  
    TXSTA1 = 0b00100100;    // BRGH = 1
  }     
  RCSTA1 = 0; 
  i = RCREG1;
  i = RCREG1;
  RCSTA1 = 0b10010000; 
  //-------------------------------------------------------
  if(uspeed2 < 3)
  {
    if(uspeed2 == 0) SPBRG2 = 194; // "3200"  
    if(uspeed2 == 1) SPBRG2 = 129; // "4800"  
    if(uspeed2 == 2) SPBRG2 = 64;  // "9600"  
    TXSTA2 = 0b00100000;    // BRGH = 0
  }
  else
  {
    if(uspeed2 == 3) SPBRG2 = 129; // "19200"  
    if(uspeed2 == 4) SPBRG2 = 64;  // "38400"  
    if(uspeed2 == 5) SPBRG2 = 42;  // "57600"  
    if(uspeed2 == 6) SPBRG2 = 25;  // "96000"  
    if(uspeed2 == 7) SPBRG2 = 21;  // "115200"  
    TXSTA2 = 0b00100100;    // BRGH = 1
  }            
  RCSTA2 = 0;
  i = RCREG2;
  i = RCREG2;
  RCSTA2 = 0b10010000;
}
//-----------------------------------------------------------------------------









