/******************************************************************************
  RomanLCD.c       Open-source - Oct 2009 - www.RomanBlack.com
  These are my "minimalist" functions to drive a text LCD
  with a PIC. Check the PIC pin defines and change as needed.
  
  LCD pins are setup for EasyPIC5/6;
    RB0-3 = D4-7
    RB4 = RS
    RB5 = E

******************************************************************************/


// define char LCD size
#define  RLCD_LCD_HEIGHT        2   // char LCD display lines; 1,2,4, allowed
#define  RLCD_LCD_WIDTH         16   // char LCD width; 8,12,16,20,32,40 allowed

#define  RCMD  0    // used in RomanLCD_lcd_byte(), don't change these!
#define  RCHAR 1

unsigned char RLCD_trashdata; // used inside functions


//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


// wrap compiler inline delay as a function to save code size.
void delay_6mS(void)
{
  Delay_ms(6);
}
// wrap compiler inline delay as a function to save code size.
void delay_100uS(void)
{
  Delay_us(100);
}
void delay_20uS(void)
{
  Delay_us(20);
}




//=============================================================================
//  RomanLCD_BYTE            sends a BYTE to text LCD
//=============================================================================
void RomanLCD_Byte(unsigned char RLCD_char, unsigned char RLCD_cmd)
{
  //-----------------------------------------------------
  // this sends 2 4bit nibbles to the LCD, and each
  // nibble is then clocked by a \ edge of the LCD E pin.
  //-----------------------------------------------------
           
  // send top nibble first
  RLCD_trashdata = (RLCD_char >> 4);        // get top 4 bits
  RLCD_trashdata.F5 = 1;                    // set E pin hi
  if(RLCD_cmd) RLCD_trashdata.F4 = 1;       // set RS (command=0/char=1)
  PORTB = RLCD_trashdata;                   // send data and E hi
  Delay_us(10);
  RLCD_trashdata.F5 = 0;                    // set E pin lo
  PORTB = RLCD_trashdata;                   // send data and E \ clock pulse
  Delay_us(10);
  
  // now send bottom nibble
  RLCD_trashdata = (RLCD_char & 0x0F); 
  RLCD_trashdata.F5 = 1;                                
  if(RLCD_cmd) RLCD_trashdata.F4 = 1; 
  PORTB = RLCD_trashdata;                
  Delay_us(10);
  RLCD_trashdata.F5 = 0;                        
  PORTB = RLCD_trashdata;        
  Delay_us(40);

}
//-----------------------------------------------------------------------------


//=============================================================================
//  RomanLCD_MOVE        moves LCD cursor to line 0-3 , col 0-39
//=============================================================================
void RomanLCD_Move(unsigned char RLCD_line, unsigned char RLCD_col)
{
  //-----------------------------------------------------
  // formats line, col into the 1byte command needed
  // by the LCD and send it to the LCD.
  //  LCD command; 1aaaaaaa
  //
  // This should work for all char LCD sizes;
  //  LCD display HEIGHT; 1, 2, 4
  //  LCD display WIDTH; 8, 12, 16, 20, 32, 40
  //-----------------------------------------------------
  
  // safe limit to the defined char LCD screen size
  if(RLCD_line >= RLCD_LCD_HEIGHT) RLCD_line = (RLCD_LCD_HEIGHT - 1);
  if(RLCD_col >= RLCD_LCD_WIDTH)   RLCD_col = (RLCD_LCD_WIDTH - 1);
  
  // (now re-use variable SH1_col to hold the command)
  // for 4 line displays only; if line >1 fix the address
  if(RLCD_line.F1) RLCD_col += RLCD_LCD_WIDTH;        
  
  // for 2 and 4 line displays; set address bit6 if on ODD line
  if(RLCD_line.F0) RLCD_col.F6 = 1;        
  
  // set bit7, makes this byte into an LCD move command
  RLCD_col.F7 = 1;
  
  // finally send that command to the LCD
  RomanLCD_Byte(RLCD_col,RCMD);

}
//-----------------------------------------------------------------------------


//=============================================================================
//  RomanLCD_OUT       send text string to COG LCD 
//=============================================================================
void RomanLCD_Out(unsigned char sline, unsigned char scol,
                          unsigned char *stext)
{
  //-----------------------------------------------------
  unsigned char scount;
  
  // move display cursor
  RomanLCD_Move(sline,scol);

  // now just print each char until NULL found
  scount = 0;
  while(stext[scount])
  {
    RomanLCD_Byte(stext[scount],RCHAR);
    scount++;
  }
}
//-----------------------------------------------------------------------------

//=============================================================================
//  RomanLCD_CHAR       send one text character to text LCD 
//=============================================================================
void RomanLCD_Char(unsigned char schar)
{
  //-----------------------------------------------------
  RomanLCD_Byte(schar,RCHAR);
}
//-----------------------------------------------------------------------------



//=============================================================================
//  RomanLCD_INIT     initialise COG LCD 
//=============================================================================
void RomanLCD_Init(void)
{
  //-----------------------------------------------------
  // The LCD operates in 4bit data mode. (6 wires)
  // LCD pin setup below is for EasyPIC5/6
  //-----------------------------------------------------
  
  // manually send first control nibble to LCD!
  PORTB = 0b00100010;         // 0010 code for init, E=1, RS=0
  delay_100uS();
  PORTB = 0b00000010;         // make E low, is LCD \ clock pulse
  delay_6mS();                // 6mS delay needed by LCD
  
  //PORTB = 0b00100010;         // repeat, needed for some OLD displays        
  //delay_100uS();
  //PORTB = 0b00000010;         
  //delay_6mS();                
  
  //-----------------------------------------------------
  // now we can send commands to LCD as bytes
  //-----------------------------------------------------
  
  // Cmd, set interface length (function set)
  RomanLCD_Byte(0x28,RCMD);     // 2=4bit, 8=2line display (I use this standard)
  //RomanLCD_Byte(0x20,RCMD);     // 2=4bit, 0=1line display (was testing!)
  delay_6mS();
  
  // Cmd, display on/off control
  RomanLCD_Byte(0x0C,RCMD);     // disp on, cursor off, block off.
  delay_100uS();
  
  // Cmd, clear display (is the slow command)
  RomanLCD_Byte(0x01,RCMD);     // clear display
  delay_6mS();                  // 6mS is long enough even for old LCDs
  
  // Cmd, entry mode set
  RomanLCD_Byte(0x06,RCMD);     // 6=move cursor right, but don't scroll display
  delay_6mS();
}
//-----------------------------------------------------------------------------







