/******************************************************************************
  SH1_Counter.c   A dual 16-digit up-counter
  (requires;) #include "Shift1_LCD.c"
  Open-source  -  29th Nov 2009  -  www.RomanBlack.com

  This uses a 16x2 LCD to act as 2 independant 16-digit counters.
  If GP5 is held low, the bottom counter is not used. This means
  it can be used on a 16x1 LCD as a single 16-digit counter.
  The total dual-count and display 32 chars takes about 120mS
  so the maximum count frequency is around 5 counts per
  second, maybe less depending on input duty cycle.
  (if clocking 1 counter only, that is faster; 10 counts per second)
  The debounce requires that a count input is low >7mS to count.

  PIC pins;
   GP0  counter 1 signal in, inc counter1 on \ edge
   GP1  counter 2 signal in, inc counter1 on \ edge
   GP2  (data out to Shift1-LCD)
   GP3  low = reset counter 1 (needs 10k pullup resistor)
   GP4  low = reset counter 2
   GP5  low = 1 counter, hi (default) = 2 counters
******************************************************************************/


// global vars

unsigned char count1[17];   // holds the 16 digits counter values (as text!)
unsigned char count2[17];   // 

unsigned char i;            // used in loops etc
unsigned char debounce;     // for debouncing the count inputs


#define PIN_COUNT1    GPIO.F0    // 
#define PIN_COUNT2    GPIO.F1    // 
#define PIN_RESET1    GPIO.F3    // low = pressed (needs 10k pullup resistor)
#define PIN_RESET2    GPIO.F4    // low = pressed
#define PIN_SINGLE    GPIO.F5    // 

//-----------------------------------------------------------------------------
// this line adds my routines for Shift1 LCD driving
#include "Shift1_LCD.c"
//-----------------------------------------------------------------------------


//=============================================================================
//   CLEAR COUNTER
//=============================================================================
void clear_counter(unsigned char *ccount)
{
  //-----------------------------------------------------
  // since the counter values are stored as text, just
  // fill the text string with blanks and a '0'
  for(i=0; i<15; i++)
  {
    ccount[i] = ' ';  // fill counter with blanks
  }
  ccount[15] = '0';
  ccount[16] = 0;
}
//-----------------------------------------------------------------------------


//=============================================================================
//   INC COUNTER
//=============================================================================
void inc_counter(unsigned char *icount)
{
  //-----------------------------------------------------
  // the counter values are stored as text.
  // so inc them using a special procedure starting
  // at the final (right) character then working back.
  //-----------------------------------------------------
  // loop and inc the counter
  i = 15;
  while(i < 16)
  {
    if(icount[i] < '0') icount[i] = '0'; // change digit to '0' before inc it
    icount[i]++;                         // inc the digit
  
    if(icount[i] <= '9') break;          // if no digit roll, inc is done
    icount[i] = '0';                     // digit roll, so clr it, and inc next
    i--;
  }
  // check for final roll and do a counter reset, to put blanks back in
  if(icount[0] == '0') clear_counter(icount);
}
//-----------------------------------------------------------------------------


// wrap this delay in a function to save ROM
void Delay_mS_400(void)
{
   Delay_mS(400);
}

//=============================================================================
//   MAIN
//=============================================================================
void main ()
{
  //-----------------------------------------------------
  // PIC 12F675  setup ports

  ANSEL = 0;            // ADC off
  CMCON = 0x07;         // comparators off
  GPIO =   0b00000100;  // GP2 high default for Shift1 out
  TRISIO = 0b00111011;  // GP2 is Shift1 out, rest are inputs
  WPU =    0b00111011;  // pin pullups; 1 = pullup on

  //-----------------------------------------------------
  // setup timers
  T1CON = 0b00100001;   // TMR1 on, 1:4 prescale, TMR1H is about 1mS per tick
  
  // setup vars
  clear_counter(count1);
  clear_counter(count2);
  debounce = 0;

  //-----------------------------------------------------
  // initialise LCD
  Delay_mS_400();
  SH1_Lcd_Init();           // init LCD
  SH1_lcd_backlight = 0;    // Shift1-LCD backlight OFF
  Delay_mS_400();

  // display counter start values
  SH1_Lcd_Out(0,0,count1);
  if(PIN_SINGLE) SH1_Lcd_Out(1,0,count2);
  
  //-----------------------------------------------------

  // main run loop here
  while(1)
  {
    //-------------------------------------
    // simple debounce counter
    TMR1L = 0;
    TMR1H = 0;
    while(TMR1H < 1); // waits for about 1mS

    //-------------------------------------
    // check counter 1
    
    // inc counter 1 if passes debounce test
    if(PIN_COUNT1) debounce = (debounce & 0xF0);   // clear count1 debounce bits
    if(!PIN_COUNT1)
    {
      i = (debounce & 0x0F);
      if(i < 0x08)  debounce += 0x01;
      if(i == 0x08)
      {
        inc_counter(count1);      // inc counter 1
        SH1_Lcd_Out(0,0,count1);
        debounce += 0x01;
      }
    }

    // reset counter 1 if reset button is pressed
    if(!PIN_RESET1)
    {
      clear_counter(count1);
      SH1_Lcd_Out(0,0,count1);
    }

    //-------------------------------------
    // check and display counter 2 ONLY if it is turned on
    if(PIN_SINGLE)   // if counter 2 is ON
    {
      // inc counter 2 if passes debounce test
      if(PIN_COUNT2) debounce = (debounce & 0x0F);   // clear count2 debounce bits
      if(!PIN_COUNT2)
      {
        i = (debounce & 0xF0);
        if(i < 0x80)  debounce += 0x10;
        if(i == 0x80)
        {
          inc_counter(count2);      // inc counter 1
          SH1_Lcd_Out(1,0,count2);
          debounce += 0x10;
        }
      }
    
      // reset counter 2 if reset button is pressed
      if(!PIN_RESET2)
      {
        clear_counter(count2);
        SH1_Lcd_Out(1,0,count2);
      }    
    }
    //-------------------------------------
  }
}
//-----------------------------------------------------------------------------





