/******************************************************************************
  Shift1_LCD.c        5th Oct 2009 Roman Black - www.RomanBlack.com

  This is a library of functions to control a LCD using 1 PIC pin and
  a 74HC595 8bit shift register.
******************************************************************************/

// user constants and vars used in SH1 library
// set these values as you need for each project;
#define	SH1_XTAL	4          // xtal MHz, round down to integer (min is 4)
#define	SH1_PIN 	GPIO.F2    // set this for your PIC output pin
#define SH1_LCD_HEIGHT 2     // char LCD display lines; 1,2,4, allowed
#define SH1_LCD_WIDTH	 16    // char LCD width; 8,12,16,20,32,40 allowed

// used in SH1_Lcd_Byte
#define  SCMD  0             // don't touch these
#define  SCHAR 1

// user variable, controls the LCD backlight
unsigned char SH1_lcd_backlight;	// 1 = LCD backlight ON

// private vars used within functions
unsigned char SH1_trashdata;		  // don't touch this

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


//=============================================================================
//  SH1_DELAY_6mS     (wrapped in a function to save code size)
//=============================================================================
void SH1_Delay_6mS(void)
{
	Delay_mS(6);
}
//-----------------------------------------------------------------------------


//=============================================================================
//  SH1 DELAY 15uS     (wrapped in a function to save code size)
//=============================================================================
void SH1_Delay_15uS(void)
{
    // with 4MHz xtal takes 2uS to get here, 2uS to return
    // so delay needs to be 11uS
    Delay_uS(11);
}
//-----------------------------------------------------------------------------


//=============================================================================
//  SH1_SEND_BYTE
//=============================================================================
void SH1_Send_Byte(unsigned char SH1_sdata)
{
	//-----------------------------------------------------
	// Sends one byte out to the 74HC595 shift register
	// using RomanBlack's Shift1 protocol. This is a timed
	// protocol, using 1 PIC pin and 3 timed LO/HI periods to
	// send 3 type of pulse.
	//
	// The byte to be sent is "SH1_sdata";
	//  bit 4-7; LCD 4bit data
	//  bit 3; LCD E
	//  bit 2; LCD RS
	//  bit 1; (optional) LO = LCD backlight ON
	//  bit 0; (ignored) is always zero
	//-----------------------------------------------------
	unsigned char SH1_bit_count;	// used to send 8 bits

	// automatically handle the LCD backlight
	SH1_sdata.F1 = 1;		  // default, LCD backlight is OFF
	if(SH1_lcd_Backlight) SH1_sdata.F1 = 0;  // or turn backlight ON
	
  // now send 7 bits
	SH1_bit_count = 7;
	while(SH1_bit_count)
	{
		if(SH1_sdata.F7)	// if HI bit, make 1uS low pulse
		{
			SH1_PIN = 0;	// make LO pulse
      #if SH1_XTAL > 4
      {
        asm nop;
      }
      #endif
      #if SH1_XTAL > 8
      {
        asm nop;
      }
      #endif
      #if SH1_XTAL > 10
      {
        asm nop;
      }
      #endif
      #if SH1_XTAL > 12
      {
        asm nop;
      }
      #endif
      #if SH1_XTAL > 16
      {
        asm nop;
      }
      #endif
      SH1_PIN = 1;        // end LO pulse
			SH1_Delay_15uS();   // safe 15uS pulse recovery
		}
		else				// else is LO bit!
		{
			SH1_PIN = 0;		    
      SH1_Delay_15uS();   // 15uS LO pulse
			SH1_PIN = 1;		    
      SH1_Delay_15uS();	  // 30uS recovery
      SH1_Delay_15uS();	
		}
		// now we have sent that bit out using Shift1 timed protocol!
		SH1_sdata <<= 1;      // get the next bit
		SH1_bit_count--;      // send all 7 bits
	}

	// The Shift1 protocol requires that the 8th bit is very
	// long, this causes the 74HC595 shift register to latch
	// all the 8 bits to its output port.
	// NOTE! the 8th bit (bit0) will always be received as zero.
	SH1_PIN = 0;	// send 8th bit, lo pulse = 14x15 = 210uS		    
  for(SH1_bit_count=14; SH1_bit_count; SH1_bit_count--) SH1_Delay_15uS();  
	SH1_PIN = 1;	// and hi recovery 20x15 = 300 uS	    
  for(SH1_bit_count=20; SH1_bit_count; SH1_bit_count--) SH1_Delay_15uS();  
}
//-----------------------------------------------------------------------------


//=============================================================================
//  SH1_LCD_BYTE    	   sends a command or char byte to LCD
//=============================================================================
void SH1_Lcd_Byte(unsigned char SH1_byte, unsigned char SH1_cmd)
{
	//-----------------------------------------------------
	// this sends 2 4bit nibbles to the LCD, and each
	// nibble is clocked by a \ edge of the LCD E pin.
	// the 6 LCD pins are controlled by a 74HC595 shift
	// register using RomanBlack's Shift1 timed protocol.
	//-----------------------------------------------------
	// send top nibble first
	SH1_trashdata = (SH1_byte & 0xF0); // get top 4 bits
	SH1_trashdata.F3 = 1;				       // set E pin hi
  if(SH1_cmd) SH1_trashdata.F2 = 1;  // set RS (command=0/char=1)
	SH1_Send_Byte(SH1_trashdata);		   // send data and E hi
	SH1_trashdata.F3 = 0;				       // set E pin lo
	SH1_Send_Byte(SH1_trashdata);		   // send data and E \ clock pulse

	// now send bottom nibble
	SH1_trashdata = (SH1_byte << 4);   // get bottom 4 bits
	SH1_trashdata.F3 = 1;				
  if(SH1_cmd) Sh1_trashdata.F2 = 1;  // set RS (command=0/char=1)
	SH1_Send_Byte(SH1_trashdata);		
	SH1_trashdata.F3 = 0;			
	SH1_Send_Byte(SH1_trashdata);	
}
//-----------------------------------------------------------------------------


//=============================================================================
//  SH1_LCD_CHAR    	sends a character byte to display on LCD
//=============================================================================
void SH1_Lcd_Char(unsigned char SH1_char)
{
	//-----------------------------------------------------
	// this sends a character to display on the LCD.
  SH1_Lcd_Byte(SH1_char,SCHAR);
}
//-----------------------------------------------------------------------------


//=============================================================================
//  SH1_LCD_MOVE    	moves LCD cursor to line 0-3 , col 0-39
//=============================================================================
void SH1_Lcd_Move(unsigned char SH1_line, unsigned char SH1_col)
{
	//-----------------------------------------------------
	// formats line, col into the 1byte command needed
	// by the LCD and send it to the LCD.
	//  LCD command; 1aaaaaaa
	//
	// This should work for all char LCD sizes;
	//  LCD display HEIGHT; 1, 2, 4
	//  LCD display WIDTH; 8, 12, 16, 20, 32, 40
 	//-----------------------------------------------------

	// safe limit to the defined char LCD screen size
	if(SH1_line >= SH1_LCD_HEIGHT) 	SH1_line = (SH1_LCD_HEIGHT - 1);
	if(SH1_col >= SH1_LCD_WIDTH) 	SH1_col = (SH1_LCD_WIDTH - 1);

	// (now re-use variable SH1_col to hold the command)
	// for 4 line displays only; if line >1 fix the address
	if(SH1_line.F1) SH1_col += SH1_LCD_WIDTH;	

	// for 2 and 4 line displays; set address bit6 if on ODD line
	if(SH1_line.F0) SH1_col.F6 = 1;	

	// set bit7, makes this byte into an LCD move command
	SH1_col.F7 = 1;

	// finally send that command to the LCD
	SH1_Lcd_Byte(SH1_col,SCMD);
}
//-----------------------------------------------------------------------------


//=============================================================================
//  SH1_LCD_OUT       send text string to LCD 
//=============================================================================
void SH1_Lcd_Out(unsigned char sline, unsigned char scol,
                          unsigned char *stext)
{
  //-----------------------------------------------------
  unsigned char scount;
  
  // move display cursor
  SH1_Lcd_Move(sline,scol);

  // now just print each char until NULL found
  scount = 0;
  while(stext[scount])
  {
    SH1_Lcd_Char(stext[scount]);
    scount++;
  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//  SH1_LCD_INIT     initialise text LCD 
//=============================================================================
void SH1_Lcd_Init(void)
{
	//-----------------------------------------------------
	// The text LCD operates in 4bit data mode. (6 wires)
	// It is controlled from 1 PIC pin using RomanBlack's
	// Shift1 protocol and a 74HC595 shift register.
	//-----------------------------------------------------
	// manually send first control nibble to LCD!
	SH1_Send_Byte(0b00101000);   // 0010 code for init, E=1, RS=0
	SH1_Send_Byte(0b00100000);   // make E low, is LCD \ clock pulse
	SH1_Delay_6mS();				     // 6mS delay needed by LCD

	//-----------------------------------------------------
	// now we can send commands to LCD as bytes
	//-----------------------------------------------------
	// Cmd, set interface length
	SH1_Lcd_Byte(0x28,SCMD);		 // 2=4bit, 8=2line display
	//SH1_Lcd_Byte(0x20,SCMD);		 // 2=4bit, 0=1line display
	SH1_Delay_6mS();
	//SH1_Lcd_Cmd(0x28);	  // (some older LCDs need to do it twice)
	// SH1_delay_6mS();

	// Cmd, display on/off control
	SH1_Lcd_Byte(0x0C,SCMD);	   // 0x0C disp on, cursor off, block off.
	SH1_Delay_6mS();
	
	// Cmd, clear display (is the slow command)
	SH1_Lcd_Byte(0x01,SCMD);     // clear display
	SH1_Delay_6mS();		         // 12mS is long enough even for old LCDs
	SH1_Delay_6mS();

	// Cmd, entry mode set
	SH1_Lcd_Byte(0x06,SCMD);		 // 6=move cursor right, but don't scroll display
	SH1_Delay_6mS();
}
//-----------------------------------------------------------------------------



