/******************************************************************************
  SH1_Morse.c   a Morse code sound and visual trainer
  (requires;) #include "Shift1_LCD.c"
  Open-source  -  29th Nov 2009  -  www.RomanBlack.com

  This is a visual+audio morse code trainer. It plays random
  3 letter words, or sometimes a random letter or number.
  The morse speed is in standard units and complies with the
  Paris timing standard. The speed can be set in actual WPM
  from 5 WPM to 35 WPM. Ouput audio is at 750Hz.
  The extra time T can be set at extra 0, 0.5 or 1 T extra pause
  between each character.

  The LCD top line shows the 3 letter word which is repeated 4 times.
  A new random word is played after 4 seconds.
  The bottom LCD line shows the selected WPM, and the selected T.

   GP0 - button, WPM UP, lo=pressed
   GP1 - button; WPM DOWN, lo=pressed
   GP2 - (out to Shift1-LCD)
   GP3 - button; Change T, lo=pressed (needs 10k pullup ressstor)
   GP4 - morse code 750Hz out to speaker, use 500 ohm pot and 180 ohm resistor.
   GP5 - (not used)

******************************************************************************/

#define XTAL_MHz  4000000    // 4 MHz internal osc

#define PIN_BUT_WPMU  GPIO.F0   // low = pressed
#define PIN_BUT_WPMD  GPIO.F1   // low = pressed
#define PIN_BUT_T     GPIO.F3   // low = pressed (needs 10k pullup resistor)

#define PIN_MORSE_OUT GPIO.F4   // 

unsigned char wpm_tpulse;       // length of Morse T in 750Hz pulses
unsigned char wpm;              // actual wpm
unsigned char t;                // extra "t" time setting

unsigned char random;       // for playing random letter etc

unsigned char randH;		    // 16bit number used in RNG
unsigned char randL;		    // 

unsigned char wpointer;     // used to play a word from ROM

unsigned char i;            // used in loops etc
unsigned char j;            // used in loops etc

unsigned char txt[4];       // for number display

// note this WPM uses official morse timing; t = 1200 / WPM
// we express t in cycles of 750Hz, so 20 WPM; t=60mS, cycles=45
// so can use this formula; cycles = 900 / WPM
// the WPM table is 31 values, from 5 WPM to 35 WPM
const unsigned char wpm_table[31] = {180,150,129,112,100,     // 5 to 9 WPM
                                     90, 82, 75, 69, 64,
                                     60, 56, 53, 50, 47,
                                     45, 43, 41, 39, 37,
                                     36, 35, 33, 32, 31,
                                     30, 29, 28, 27, 26, 25}; // 30 to 35 WPM

//-----------------------------------------------------------------------------
// add the Morse table and Words table
#include "Morse_Table.c"
//-----------------------------------------------------------------------------
// this line adds my routines for Shift1 LCD driving
#include "Shift1_LCD.c"
//-----------------------------------------------------------------------------


//=============================================================================
//  MAKE PERIOD     makes a timed period; dot = 1, dah = 3, beep =on/off 
//=============================================================================
void make_period(unsigned char plength, unsigned char pbeep)
{
  //-----------------------------------------------------
  // this makes a period defines in Morse standard length t,
  //  DOT = period 1, beep on 
  //  DAH = period 3, beep on 
  //  interpause = period 1, beep off 
  //  char pause = period 3, beep off 
  //  word pause = period 7, beep off 
  //
  // the length of a standard period t is defined in 750Hz pulses.
  // so 20 wpm = 60mS = 45 pulses
  //-----------------------------------------------------
  unsigned char wpcount; 
  
  // loop and make the period
  while(plength)
  {
    // make one t period
    wpcount = wpm_tpulse;  // how many 750Hz pulses in t period
    TMR1L = (128-81);
    while(wpcount)
    {
      while(!TMR1L.F7);   // wait for 1/2 of a 750Hz pulse
      if(pbeep) PIN_MORSE_OUT = 1;  // speaker pulse HI    
      TMR1L -= 81;
    
      while(!TMR1L.F7);   // wait for 1/2 of a 750Hz pulse
      PIN_MORSE_OUT = 0;  // speaker pulse LO    
      TMR1L -= 81;
  
      wpcount--;
    }
    plength--;
  }
}
//-----------------------------------------------------------------------------

//=============================================================================
//  PLAY MORSE CHAR       "plays" 1 char as morse code 
//=============================================================================
void play_morse_char(unsigned char mlen)
{
  //-----------------------------------------------------
  // this plays one char as morse code.
  // we have 36 chars in MT[]; 26 aplpha and 10 numbers
  // need to convert ascii char to the right value for
  // lookup in morse table MT.
  //-----------------------------------------------------
  unsigned char mdata;

  // convert ascii to morse table value
  if(mlen > 96) mlen -= 32;   // convert lower case to upper case
  if(mlen > 64) mlen -= 65;   // is ascii, adjust for morse table
  else          mlen -= 22;   // is number, so '0' 48->26
  if(mlen > 35) mlen = 35;    // safe! stops table overflow crash
  
  // get the morse code dots from MT table
  mdata = MT[mlen];
  
  // find the length (in morse dots), length is last 3 bits
  mlen = (mdata & 0b00000111);

  // now play that morse char
  while(mlen)
  {
    if(mdata.F7) make_period(3,1);    // play a DAH  
    else         make_period(1,1);    // else play a DIT
    make_period(1,0);                 // make 1 quiet period
    mdata = (mdata << 1);             // get next dot
    mlen--;
  }

  // add 2 more quiet periods at the end, =3 periods character space
  make_period(2,0);             

  //-----------------------------------------------------
  // the character has been played with proper morse timing.
  // but here we can add an extra time; 0.5t or 1.0t
  // this small extra pause between chars makes it easier
  // for beginners.
  
  if(t)   // if extra t is selected
  {
    mlen = wpm_tpulse;              // default; extra 1.0t
    if(t == 1) mlen = (mlen >> 1);  // or extra 0.5t
    TMR1L = (128-81);
    while(mlen)
    {
      while(!TMR1L.F7);   // wait for 1/2 of a 750Hz pulse
      TMR1L -= 81;
      while(!TMR1L.F7);   // wait for 1/2 of a 750Hz pulse
      TMR1L -= 81;
      mlen--;
    }
  }
}
//-----------------------------------------------------------------------------



//=============================================================================
//  MAKE_ROM_WORD       send word from ROM to LCD 
//=============================================================================
void make_ROM_word(void)
{
  //-----------------------------------------------------
  unsigned char rtemp;
  
  // now just print each char until NULL found
  // ALSO plays each character as morse code
  while(1)
  {
    rtemp  = Words[wpointer];   // gets the character from ROM table
    if(rtemp == '.') break;
    play_morse_char(rtemp);     // play char as morse
    SH1_Lcd_Char(rtemp);        // then draw char on LCD
    wpointer++;
  }

  // get here after a word was played.
  // so add 4 more periods, this =7 periods for word spacing
  make_period(4,0);
}
//-----------------------------------------------------------------------------

//=============================================================================
//   DELAY_MS_250      wrap this delay in a function to save ROM
//=============================================================================
void Delay_mS_250(void)
{
  //-----------------------------------------------------
  // makes a 200mS delay using TMR1H, a TMR1H tick is about 2mS 
  //-----------------------------------------------------
  TMR1H = 0;
  while(TMR1H < 125);
}
//-----------------------------------------------------------------------------


//=============================================================================
//   MAKE RANDOM BYTE
//=============================================================================
void make_random_byte(void)
{
  //-----------------------------------------------------
  // this function make 8 new psuedo random bits in randL.
	// 16bit random number generator is taken from Microchip's
	// AN544 math routines appnote "psuedo random number generator";
	//  1. XOR bits 15^14 into A
	//  2. XOR bits 12^3 into B
	//  3. XOR bits A^B into C
	//  4. left shift rand var 1 bit
	//  5. put C in bit 0
	//-------------------------------------------------
  unsigned char rng_bitcount;	
  unsigned char rand_temp;    

	// generate 8 new RNG bits (in randL);
	rng_bitcount = 8;
	while(rng_bitcount)
	{
		// 1. XOR bits 15^14 into A
		rand_temp = 0;
		if(randH.F7 != randH.F6) rand_temp.F2 = 1;

		// 2. XOR bits 12^3 into B
		if(randH.F4 != randL.F3) rand_temp.F1 = 1;

		// 2. XOR bits A^B into C
		if(rand_temp.F2 != rand_temp.F1) rand_temp.F0 = 1;

		// 4. left shift rand var 1bit
		asm CLRF STATUS			;
		asm RLF randL,f			;
		asm RLF randH,f			;

		// 5. put C in bit 0
		if(rand_temp.F0) randL++;

		rng_bitcount--;
	}
}
//-----------------------------------------------------------------------------


//=============================================================================
//   PICK RANDOM WORD
//=============================================================================
void pick_random_word(void)
{
  //-----------------------------------------------------
  // this function does 2 things;
  // 1. uses var "random" to pick one of the available words
  // 2. returns with the pointer to the first char of that word
  // in variable "wpointer"
  //-----------------------------------------------------
  unsigned char wrand;
  unsigned char wnull;
  unsigned char w;
  
  // 1. pick one word at random
  wrand = randL;   // get random
  while(wrand >= NUM_WORDS)   // loop until a legal word is found
  {
    wrand -= NUM_WORDS;
  }         

  // now we have a legal word number 0-x in wrand
  // 2. find the first char of that word
  wpointer = 0;
  wnull = 0;
  w = 0;
  while(1)    // loop forward a word at a time
  {
    while(Words[w] != '.') w++;  // loop forward to find a STOP char  
    // found a STOP!
    if(wnull == wrand) break;   // done!
    wnull++;                    // else count another word++
    w++;                        // ready for next char
    wpointer = w;               // get the pointer to the next word
  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//   SH1 ROM OUT      display a text string from ROM to LCD bottom line
//=============================================================================
void SH1_Rom_Out(unsigned char x, const unsigned char *romstring)
{
  //-----------------------------------------------------
  // move to the colum ready to write to LCD
  SH1_Lcd_Move(1,x);

  // loop and write chars from ROM
  x = 0;
  while(1)
  {
    i = romstring[x];
    if(!i) return;
    SH1_Lcd_Char(i);
    x++;
  }
}
//-----------------------------------------------------------------------------



//=============================================================================
//   MAIN
//=============================================================================
void main()
{
  //-----------------------------------------------------
  // PIC 12F675  setup ports

  ANSEL = 0;            // no ADC used
  CMCON = 0x07;         // comparators all OFF

  GPIO =   0b00000100;  // GP2 high default for Shift1 out
  TRISIO = 0b00101011;  // GP2 is Shift1 out, GP0 is freq out, GP0,1,3 buttons
  WPU =    0b00101011;  // pin pullups; 1 = pullup on (for button)

  //-----------------------------------------------------
  // laod the calibration value for internal RC osc
  OSCCAL = OSCCAL;    // compiler needs this line to recognise OSCCAL
  asm {
    BSF STATUS, RP0   ; Bank 1
    CALL 0x03FF       ; Get the cal value
    MOVWF OSCCAL      ; Calibrate
  }
  
  //-----------------------------------------------------
  // TMR0
  OPTION_REG = 0b00000111;  // pullups ON, TMR0 prescaler 1:256
  
  // TMR1
  T1CON = 0b00110001;   // 1:8 prescale, each TMR1H tick =2.048mS (approx 1mS)
  
  //-----------------------------------------------------
  // initialise LCD
  Delay_mS_250();
  SH1_Lcd_Init();           // init LCD
  SH1_lcd_backlight = 0;    // Shift1-LCD backlight OFF

  SH1_Lcd_Out(0,0,"Morse");   // start message (optional)

  // setup variables etc
	randH = 0x30;		// put seed in RNG
	randL = 0x45;		// put seed in RNG
  //random = (random ^ wpm_tpulse);

  // set the Morse speed in words per minute
  wpm = 20;
  wpm_tpulse = 45;
  SH1_Rom_Out(1,"WPM 20");
  t = 2;
  SH1_Rom_Out(9,"T+ 1.0");
  

  //-----------------------------------------------------
  // main run loop here
  while(1)
  {
    //-------------------------------------------
    // 4 second delay, and check buttons during delay
    for(i=0; i<16; i++)
    {
      Delay_mS_250();     // 250mS between each button press 

      // check WPM adjust buttons
      if(!PIN_BUT_WPMU || !PIN_BUT_WPMD)    // if either is pressed
      {
        if(!PIN_BUT_WPMU && wpm < 35) wpm++;    // change WPM UP
        if(!PIN_BUT_WPMD && wpm > 5)  wpm--;    // change WPM DOWN
        wpm_tpulse = wpm_table[wpm-5];  // load the WPM t period from table
        i = wpm;                        // then display the WPM
        j=0;
        while(i >= 10)
        {
          i -= 10;
          j++;
        }
        Sh1_Lcd_Move(1,5);
        Sh1_Lcd_Char(j + '0');
        Sh1_Lcd_Char(i + '0');
        i = 0;
      }
      // check t adjust button
      if(!PIN_BUT_T)    // if pressed
      {
        t++;                            
        if(t > 2) t = 0;                // change t
        if(t == 0) SH1_Rom_Out(12,"  0");
        if(t == 1) SH1_Rom_Out(12,"0.5");
        if(t == 2) SH1_Rom_Out(12,"1.0");
        i = 0;
      }
    }
    
    //-------------------------------------------
    //clear top line of display
    SH1_Lcd_move(0,0);
    for(i=0; i<16; i++) SH1_Lcd_Char(' ');  

    // next decide whether to play a word, letter or a number
    make_random_byte();
    if(randL.F7)    // if its a word (50% chance)
    {
      SH1_Lcd_Move(0,0);         
      i=4;                            // play word 4 times
      while(i)
      {
        pick_random_word();           // get pointer to word
        make_ROM_word();              // play the word (and display it)
        SH1_Lcd_Char(' ');
        i--;
      }
    }
    else    // else its a letter or number
    {
      random = randL;               
      SH1_Lcd_Move(0,0);
      if(randL.F6)    // if its a letter (25% chance)
      {
        while(random >= 26) random -=26;  // get random letter 0-25
        random += 'a';                  // make it a text letter
        i=4;                            // play letter 4 times
        while(i)
        {
          play_morse_char(random);      // play the number
          SH1_Lcd_Char(random);         // and display it
          SH1_Lcd_Char(' ');
          i--;
        }
      }
      else    // else its a number (25% chance)
      {
        while(random >= 10) random -=10;  // get a random number 0-9
        random += '0';                  // make it a text number
        i=4;                            // play number 4 times
        while(i)
        {
          play_morse_char(random);      // play the number
          SH1_Lcd_Char(random);         // and display it
          SH1_Lcd_Char(' ');
          i--;
        }
      }
    }
    
    //-------------------------------------------

  }
}
//-----------------------------------------------------------------------------








