/******************************************************************************
  SH1_Servo.c     28th Oct 2009 Roman Black
  Shift1_LCD.c is a library of functions to control a LCD using
  1 PIC pin and a 74HC595 8bit shift register.

  This project drives an RC servo under control from a pot on ADC pin.
  The Servo pulse length is shown on the LCD in actual uS.

  NOTE! This application uses a special function to write to the LCD
  that makes sure that TMR1 interrupt will NOT occur when writing
  to the LCD!!

******************************************************************************/

// Global variables
unsigned int adc_pot absolute 0x20;     // ADC "voltage" 10bit value
unsigned char adc_potL absolute 0x20;   // and allow access to each byte
unsigned char adc_potH absolute 0x21;   // by overloading variables

unsigned char range;            // adjust range; 1uS resolution or 2uS
#define RANGE_FINE   0
#define RANGE_COARSE 1

unsigned char servopulse;       // servo pulse period HIGH or LOW
unsigned char servo_msb;        // servo 16bit pulse period to go into TMR1
unsigned char servo_lsb;        //
#define PIN_SERVO      GPIO.F1
#define PIN_BUT_RANGE  GPIO.F3

unsigned int microsec;          // microsecond period of servo HIGH pulse
unsigned int math16;            // used only for calcs

unsigned char txt[6];           // used to hold text string to display


//-----------------------------------------------------------------------------
// My routines for doing Shift1 protocol LCD driving.
// NOTE! check the LCD values in this file and edit if needed;
#include "Shift1_LCD.c"
//-----------------------------------------------------------------------------

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void  interrupt(void)
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
{
	//----------------------------------------------------
	// is TMR1 overflow int!
	//
	// Each of the 8 servos have a 16bit variable which represents
	// a direct TMR1 value. If TMR1 is loaded with this value,
	// it will be the correct time for that servo's hi pulse (1mS to 2mS).
	// so the value is really (0 - time) so the TMR1 roll int will occur
	// when we need to end the hi pulse.
	// since this is just for the TalkBot slave, we assume that the
	// 8 pins are all dedicated outs.
	//----------------------------------------------------

  // set the servo pulse LOW first
  PIN_SERVO = 0;

  // then turn TMR1 off so we can write to it
  T1CON = 0;  

  // check if next period is HIGH or LOW pulse
  if(servopulse)   // if HIGH
  {
      // write the high pulse period to TMR1
			TMR1H = servo_msb;
			TMR1L = servo_lsb;

			// TMR1 back on amd start the HIGH pulse
      T1CON = 0b00010001;		// TMR1 ON, at 1:2 prescale
      PIN_SERVO = 1;
      servopulse = 0;
  }
  else     // else is LOW
  {                                               
      // write the low pulse period to TMR1
			// just use 18mS low period for simplicity;
      // so use (65536uS - 18000uS) = 47536 = 0x B9 B0
      TMR1H = 0xB9;
			TMR1L = 0xB0;
  
			// TMR1 back on amd start the LOW pulse
      T1CON = 0b00010001;		// TMR1 ON, at 1:2 prescale
      servopulse = 1;
  }

	//----------------------------------------------------
	// clear the TMR1 overflow int and exit
	PIR1.TMR1IF = 0;
}
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


//=============================================================================
//  TEST ADC
//=============================================================================
void test_adc(void)
{
  //-------------------------------------------------------
  // read battery voltage from AN0 (RA0)
  ADCON0.GO = 1;      // start ADC conversion
  while(ADCON0.GO);   // wait for conversion to be finished

  adc_potL = ADRESL;  // save bottom 8 bits
  adc_potH = ADRESH;  // and top 2 bits
}
//-----------------------------------------------------------------------------


//=============================================================================
//  SH1_LCD_TMR1INT_OUT       send text string to Shift1-LCD 
//=============================================================================
void SH1_Lcd_TMR1Int_Out(unsigned char sline, unsigned char scol,
                          unsigned char *stext)
{
  //-----------------------------------------------------
  // NOTE!! this is a special function to make sure that
  // writing to the Shift1-LCD will NOT occur at the same
  // time as a TMR1 interrupt.
  //
  // Each byte we write to the LCD (both char and cmd bytes)
  // will take approx 3.5 to 4mS. So what we do is check that
  // the TMR1 interrupt is at least 5mS away, then if so,
  // it is ok to send a byte. If not, we just wait until
  // it is safe.
  // Since TMR1 is operating at 1MHz, TMR1H full = 65mS
  // before an int can occur.
  // So if TMR1H < 235 then there must be at least 5mS before int.
  // Using 230 to allow a little more safety margin.
  //-----------------------------------------------------
  unsigned char scount;
  
  // move display cursor
  while(TMR1H >= 230);   // wait here until TMR1 interrupt is safe
  SH1_Lcd_Move(sline,scol);

  // now just print each char until NULL found
  scount = 0;
  while(stext[scount])
  {
    while(TMR1H >= 230);   // wait here until TMR1 interrupt is safe
    SH1_Lcd_Char(stext[scount]);
    scount++;
  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//  text messages here as functions to save RAM
//=============================================================================
void message_coarse(void)
{
  SH1_Lcd_TMR1Int_Out(0,0,"Coarse");
}
void message_fine(void)
{
  SH1_Lcd_TMR1Int_Out(0,0,"Fine  ");
}


//=============================================================================
//  MAIN
//=============================================================================
void main()
{
  //-------------------------------------------------------
  // 12F675 setup registers etc
  
  ANSEL =  0b00100001;   // ADC fosc32, AN0 is analog input
  ADCON0 = 0b10000001;   // right justified, VREF=Vdd, ADC=AN0, ADC is ON

  CMCON = 0x07;   // comparators OFF

  GPIO =   0b00000100;    // GP2 normally HI, goes to Shift1-LCD
  TRISIO = 0b00001001;    // GP2 is Shift1 out, GP1 is servo signal out
                          //  GP3 is range button, GP0 is ADC in
  WPU =    0b00000000;    // weak pull-ups 0=OFF

  //-------------------------------------------------------
  // setup TMR1 to run at 1MHz (1uS per timer tick)
  T1CON = 0b00010001;   // TMR1 ON, at 1:2 prescale

  // setup variables etc as needed
  range = RANGE_FINE;

  //-------------------------------------------------------
  // now setup the LCD
  Delay_mS(200);
  SH1_Lcd_Init();
  SH1_Lcd_Backlight = 0;  // 0 = backlight OFF, 1 = backlight ON
  
  // show "Fine" on LCD
  message_fine();
  Delay_mS(200);

  // finally turn TMR1 interrupt ON before main loop
  PIE1 =   0b00000001;  // TMR1IE on
  INTCON = 0b11000000;  // GIE on, PEIE on

	//-----------------------------------------------------
  // main loop here. 

  while(1)
  {
    //---------------------------------------
    // do this loop about every 0.3 secs
    Delay_mS(300);          

    // toggle range if range button is pressed (pressed = low)
    if(!PIN_BUT_RANGE)
    {
      if(range == RANGE_FINE)
      {
        range = RANGE_COARSE;
        message_coarse();
      }
      else
      {
        range = RANGE_FINE;
        message_fine();
      }
    }

    //---------------------------------------
    // read pot adc, and format the pot adc value 0-1023 to uS
    // there are 2 ranges; 1uS resolution and 2uS resolution
    test_adc();             
    if(range == RANGE_FINE)
    {
      microsec = ((1500-(1024/2)) + adc_pot);   // now is 988uS to 2011uS range for servo pulse
    }
    else   // else must be RANGE_COARSE
    {
      microsec = ((1500-1024) + (adc_pot*2));  // coarse is 494uS to 2540uS range
    }

    //---------------------------------------
    // display servo high period to LCD in uS
    wordtostr(microsec,txt);
    SH1_Lcd_TMR1Int_Out(0,8,txt);
    SH1_Lcd_TMR1Int_Out(0,14,"uS");

    //---------------------------------------
    // calc the TMR1 period for the servo high pulse    
    // this must be 2 values hi/lo bytes.
    // they aslo need to be the inverse, as TMR1 counts UP
    math16 = (microsec - 12);     // compensate for interrupt latency
    math16 = (0 - math16);        // invert period to get TMR1 16bit value!

    servo_msb = (math16 / 256);   // values ready to be put in TMR1
    servo_lsb = (math16 & 0x00FF);
  }
}
//-----------------------------------------------------------------------------


