/******************************************************************************
 * Project name:
     EP6_freq.c   EasyPIC6 only, is a freq/rpm meter using the COG display
 * Copyright:
     Open-source  -  Oct 2009  -  Roman Black
 * Description:
     I modified the EasyPIC6 COG text display example to make a simple
     frequency/rpm meter. The incoming frequency to be measured must be
     connected to PORTC.F0 (T1CKI) pin. Max freq measured is 65000 Hz.
     Note! If you don't have a freq signal connected, you can test it
     by pressing the RC0 pushbutton REALLY quickly. :)

 * Test configuration:
     MCU:             PIC16F887
                      http://ww1.microchip.com/downloads/en/DeviceDoc/41291F.pdf
     Dev.Board:       EasyPIC6
                      http://www.mikroe.com/en/tools/easypic6/
     Oscillator:      HS, 8.0000 MHz
     Ext. Modules:    - freq signal comes in on PORTC.F0
     SW:              mikroC PRO for PIC v2.50
                      http://www.mikroe.com/en/compilers/mikroc/pro/pic/

******************************************************************************/

// global declarations

// Port Expander module connections
sbit  SPExpanderCS  at RA2_bit;
sbit  SPExpanderRST at RA3_bit;
sbit  SPExpanderCS_Direction  at TRISA2_bit;
sbit  SPExpanderRST_Direction at TRISA3_bit;
// End Port Expander module connections

unsigned char int_sec_count;    // used to count ints/second
unsigned char new_second;       // is set once per second

unsigned char tchar;            // text char used in LCD display numbers
unsigned int freq;              // 0-65000, holds freq value to display

unsigned long rpm;              // holds RPM for calcs AND display

char txt[12];                   // used to display number string

char sec_count;

//-----------------------------------------------------------------------------
void interrupt()
{
    // this is TMR2 overflow interrupt
    int_sec_count--;
    if(!int_sec_count)    // if reached 1 second!
    {
      // get the TMR1 count!
      T1CON = 0;        // TMR1 OFF
      freq = ((TMR1H << 8) + TMR1L);  // put TMR1 16bit value in freq
      TMR1L = 0;        // clear TMR1
      TMR1H = 0;
      T1CON = 0b00000011;   // TMR1 back ON again

      // that's everything done for this second
      new_second++;
      int_sec_count = 125;  // load ready to generate another second
    }
    TMR2IF_bit = 0;           // Clear TMR2IF before exit
}

//-----------------------------------------------------------------------------
void main()
{

  //-------------------------------------------------------
  // setup PIC 16F887 registers 
  ANSEL  = 0;                               // Configure AN pins as digital
  ANSELH = 0;
  C1ON_bit = 0;                             // Disable comparators
  C2ON_bit = 0;

  TRISC = 0b00000001;         // PORTC.F0 = input from freq signal

  //-------------------------------------------------------
  // EasyPIC6 COG text LCD setup
  // Port Expander Library uses SPI1 module
  SPI1_Init();                              // Initialize SPI module used with PortExpander

  // show startup message
  SPI_Lcd_Config(0);                        // Initialize Lcd over SPI interface
  SPI_Lcd_Cmd(_LCD_CLEAR);                  // Clear display
  SPI_Lcd_Cmd(_LCD_CURSOR_OFF);             // Turn cursor off
  SPI_Lcd_Out(1,2, "EP6  Frequency");       // display startup text to Lcd
  SPI_Lcd_Out(2,2, " & RPM  Meter ");
  Delay_1sec();
  Delay_1sec();
  
  // clear LCD again before main
  SPI_Lcd_Cmd(_LCD_CLEAR);

  //-------------------------------------------------------
  // setup the timers for frequency counting

  // setup TMR1
  T1CON = 0b00000011;       // TMR1 ON, external clock pulse on PORTC.F0

  // setup TMR2 and enable TMR2 int
  T2CON = 0b00011111;       // TMR2 ON, 8MHz xtal, 16:4:1 = 31250 Hz
  PR2 = 250;                // TMR2 int is 31250 / 250 = 125 ints/sec
  PIE1.TMR2IE = 1;          // TMR2 interrupt is on
  INTCON = 0b11000000;      // GIE=ON, PIE=ON

  // load variables ready to run
  int_sec_count = 125;

  //-------------------------------------------------------
  // now do the main run loop
  while(1)
  {                                
    // safe limit freq at 65 kHz
    if(freq > 65000) freq = 65000;

    // everytime we reach a second, calculate and display freq
    if(new_second)
    {
      new_second = 0;

      // display freq as "xxxxx Hz"
      WordToStr(freq,txt);
      SPI_Lcd_Out(1,8,txt);
      SPI_Lcd_Out(1,14,"Hz");

      // calc RPM from freq
      rpm = (freq * 60);

      // format rpm to display as "xxxxxxx RPM"
      LongToStr(rpm,txt);       // get the rpm as a string
      SPI_Lcd_Out(2,2,txt);     // and display RPM!
      SPI_Lcd_Out(2,14,"RPM");
    }
  }
}
